

import os
from enum import Enum, auto
from psycopg2 import sql


class Cohort(Enum):
    everyone = auto()
    born = auto()
    new = auto()
    reengaged = auto()
    occasional = auto()
    regular = auto()
    mgst_viewers = auto()
    first_app_open = auto()
    discover = auto()


lifecycle_cohorts = {
    Cohort.born: ('born', ),
    Cohort.new: ('new', ),
    Cohort.reengaged: ('reengaged', ),
    Cohort.occasional: ('occasional', ),
    Cohort.regular: ('regular', )
}


def read_sql(name):
    return open(os.path.join(os.path.dirname(__file__), name)).read()


def build_ret(days=[3, 7, 10, 14, 17, 21, 24, 28]):
    return {
        'ret_pageview': ",".join([f"""
            MAX(CASE WHEN t1.date = t2.date + interval '{day} day' 
                THEN 1 ELSE 0 END
            ) AS d{day}_ret
        """ for day in days]) + ',',
        'ret_pop': ','.join([f"""
            nvl(expo_pageview.d{day}_ret, 0) AS d{day}_ret
        """ for day in days]) + ',',
        'ret_agg': ','.join([f"""
            CASE WHEN SUM(CASE WHEN pop.days >= {day} THEN 1 ELSE 0 END)
                THEN SUM(d{day}_ret) :: FLOAT / 
                     SUM(CASE WHEN pop.days >= {day} THEN 1 ELSE 0 END)
                ELSE 0 END AS avg_ret_d{day:02d}
        """ for day in days]) + ','
    }


def build_query(**kwargs):
    device_join = kwargs.get('device_join', '')
    extra_metrics_query = kwargs.get('extra_metrics_query', '')
    extra_metrics_pop_select = kwargs.get('extra_metrics_pop_select', '')
    extra_metrics_pop_join = kwargs.get('extra_metrics_pop_join', '')
    extra_metrics_agg = kwargs.get('extra_metrics_agg', '')
    base_query = kwargs.get('base_query', 'sql/analysis_base.sql')

    ret_query = build_ret()

    return sql.SQL(read_sql(base_query)).format(**{
        'device_join': sql.SQL(device_join),
        'extra_metrics_query': sql.SQL(extra_metrics_query),
        'extra_metrics_pop_select': sql.SQL(extra_metrics_pop_select),
        'extra_metrics_pop_join': sql.SQL(extra_metrics_pop_join),
        'extra_metrics_agg': sql.SQL(extra_metrics_agg),
        'ret_pageview': sql.SQL(ret_query['ret_pageview']),
        'ret_pop': sql.SQL(ret_query['ret_pop']),
        'ret_agg': sql.SQL(ret_query['ret_agg']),
    })


def build_lifecycle_query(lifecycle, **kwargs):
    kwargs['device_join'] = read_sql('sql/device_join_lifecycle.sql')
    return build_query(**kwargs), {'lifecycle': lifecycle}


def build_mgst_viewers_query(**kwargs):
    kwargs['device_join'] = read_sql('sql/device_join_mgst_viewers.sql')
    return build_query(**kwargs)


def build_first_app_open_query(**kwargs):
    kwargs['device_join'] = read_sql('sql/device_join_first_app_open.sql')
    return build_query(**kwargs)


def build_discover_user_query(**kwargs):
    kwargs['device_join'] = read_sql('sql/device_join_discover.sql')
    return build_query(**kwargs)


def extras_for_chat():
    return {
        'extra_metrics_query': read_sql('sql/extra_chat_query.sql') + ', ',
        'extra_metrics_pop_select': ', ' + read_sql('sql/extra_chat_pop_select.sql'),
        'extra_metrics_pop_join': """
            LEFT JOIN chat ON chat.subject_id = exp.subject_id 
        """,
        'extra_metrics_agg': ', ' + read_sql('sql/extra_chat_agg.sql'),
    }


def extras_for_mweb():
    return {
        'extra_metrics_query': read_sql('sql/extra_mweb_query.sql') + ', ',
        'extra_metrics_pop_select': ', ' + read_sql('sql/extra_mweb_pop_select.sql'),
        'extra_metrics_pop_join': """
            LEFT JOIN app_open ON app_open.subject_id = exp.subject_id 
        """,
        'extra_metrics_agg': ', ' + read_sql('sql/extra_mweb_agg.sql'),
    }


def load_extras(name):
    return {
        'extra_metrics_query': read_sql(f'sql/{name}/query.sql') + ', ',
        'extra_metrics_pop_select': ', ' + read_sql(f'sql/{name}/pop_select.sql'),
        'extra_metrics_pop_join': read_sql(f'sql/{name}/pop_join.sql'),
        'extra_metrics_agg': ', ' + read_sql(f'sql/{name}/agg.sql'),
    }
