WITH metadata AS (
  SELECT 
    t1.uuid as experiment_id,
    max(t1.version) as experiment_version,
    max(
      GREATEST(
        date_trunc('day', t1.modified),
        date_trunc(
          'day',
          current_timestamp at time zone 'pst' - interval '90 days'
        )
      )
    ) AS start_date,
    max(
      LEAST(
        date_trunc(
          'day',
          current_timestamp at time zone 'pst' - interval '1 day'
        )
      )
    ) AS end_date
  FROM 
    dbsnapshots.minixperiment_history t1 
  WHERE 
    experiment_id = %(experiment_id)s
  GROUP BY 
    1
),
exp AS (
  SELECT 
    t1.device_id AS subject_id,
    MIN(t1.date) as date,
    MAX('control') as experiment_group,
    MAX(
      datediff(day,t1.date,m.end_date)
    ) as days 
  FROM 
    metadata m 
    JOIN mgst.experiment_device t1 ON t1.experiment_id = m.experiment_id 
    AND t1.experiment_version = m.experiment_version 
    AND t1.date >= m.start_date 
    AND t1.date <= m.end_date
    {device_join} 
  GROUP BY 
    1
),
detect_groups AS (
  SELECT 
    experiment_group,
    count(*) AS n,
    ratio_to_report(
      count(*)
    ) over() AS pct 
  FROM 
    exp 
  GROUP BY 
    1 
  HAVING 
    count(*) > 100
),
expo_fmp AS (
  SELECT 
    device_id AS subject_id,
    SUM(five_minute_play_ct) AS fmp,
    SUM(
      CASE WHEN t1.date = t2.date THEN five_minute_play_ct ELSE 0 END
    ) AS d0_fmp,
    SUM(
      CASE WHEN t1.date <= t2.date + interval '7 days' 
      AND t2.days >= 7 THEN five_minute_play_ct ELSE 0 END
    ) AS d7_fmp 
  FROM 
    mgst.expo_five_minute_play t1 
    JOIN metadata m ON t1.date >= m.start_date 
    AND t1.date <= m.end_date 
    JOIN exp t2 ON t1.device_id = t2.subject_id 
    AND t1.date >= t2.date 
  GROUP BY 
    1
),
expo_mw AS (
  SELECT 
    device_id AS subject_id,
    SUM(minute_watched_ct) AS mw,
    SUM(
      CASE WHEN lower(game_on_session_start) in %(target_games)s THEN minute_watched_ct ELSE 0 END
    ) AS mw_mobile,
    SUM(
      CASE WHEN t1.date = t2.date THEN minute_watched_ct ELSE 0 END
    ) AS d0_mw,
    SUM(
      CASE WHEN t1.date = t2.date 
      AND lower(game_on_session_start) in %(target_games)s THEN minute_watched_ct ELSE 0 END
    ) AS d0_mw_mobile,
    SUM(
      CASE WHEN t1.date <= t2.date + interval '7 day' 
      and t2.days >= 7 THEN minute_watched_ct ELSE 0 END
    ) AS d7_mw,
    SUM(
      CASE WHEN t1.date <= t2.date + interval '7 day' 
      and t2.days >= 7 
      AND lower(game_on_session_start) in %(target_games)s THEN minute_watched_ct ELSE 0 END
    ) AS d7_mw_mobile 
  FROM 
    mgst.expo_minute_watched t1 
    JOIN metadata m ON t1.date >= m.start_date 
    AND t1.date <= m.end_date 
    JOIN exp t2 ON t1.device_id = t2.subject_id 
    AND t1.date >= t2.date 
  GROUP BY 
    1
),
expo_session AS (
  SELECT 
    device_id AS subject_id,
    SUM(session_ct) AS session 
  FROM 
    mgst.expo_page_session t1 
    JOIN metadata m ON t1.date >= m.start_date 
    AND t1.date <= m.end_date 
    JOIN exp t2 ON t1.device_id = t2.subject_id 
    AND t1.date >= t2.date 
  GROUP BY 
    1
),
expo_pageview AS (
  SELECT 
    device_id AS subject_id,
    SUM(pageview_ct) AS pageview,
    SUM(
      CASE WHEN t1.date = t2.date THEN pageview_ct ELSE 0 END
    ) AS d0_pageview,
    SUM(
      CASE WHEN t1.date <= t2.date + interval '7 day' 
      and t2.days >= 7 THEN pageview_ct ELSE 0 END
    ) AS d7_pageview,
    {ret_pageview}
    MAX(CASE WHEN logged_in THEN 1 ELSE 0 END) AS logged_in,
    MAX(
      CASE WHEN t1.date <= t2.date + interval '7 day' 
      and t2.days >= 7 
      AND logged_in THEN 1 ELSE 0 END
    ) AS d7_logged_in 
  FROM 
    mgst.expo_pageview t1 
    JOIN metadata m ON t1.date >= m.start_date 
    AND t1.date <= m.end_date 
    JOIN exp t2 ON t1.device_id = t2.subject_id 
    AND t1.date >= t2.date 
  GROUP BY 
    1
),
expo_vp AS (
  SELECT 
    device_id AS subject_id,
    SUM(video_play_ct) AS vp,
    SUM(
      CASE WHEN t1.date = t2.date THEN video_play_ct ELSE 0 END
    ) AS d0_vp,
    SUM(
      CASE WHEN t1.date <= t2.date + interval '7 day' 
      and t2.days >= 7 THEN video_play_ct ELSE 0 END
    ) AS d7_vp 
  FROM 
    mgst.expo_video_play t1 
    JOIN metadata m ON t1.date >= m.start_date 
    AND t1.date <= m.end_date 
    JOIN exp t2 ON t1.device_id = t2.subject_id 
    AND t1.date >= t2.date 
  GROUP BY 
    1
),
game_follow AS (
  SELECT 
    t2.subject_id,
    COUNT(distinct game) AS game_follows,
    COUNT(
      distinct CASE WHEN LOWER(game) in %(target_games)s then game ELSE null END
    ) AS mobile_game_follows 
  FROM 
    mgst.experiment_follow_game t1 
    JOIN metadata m ON t1.date >= m.start_date 
    AND t1.date <= m.end_date 
    JOIN exp t2 ON t1.device_id = t2.subject_id 
    AND t1.date >= t2.date 
  GROUP BY 
    1
),
follow AS (
  SELECT 
    t2.subject_id,
    SUM(follow_cnt) AS follows 
  FROM 
    mgst.experiment_follow t1 
    JOIN metadata m ON t1.date >= m.start_date 
    AND t1.date <= m.end_date 
    JOIN exp t2 ON t1.device_id = t2.subject_id 
    AND t1.date >= t2.date 
  GROUP BY 
    1
),
sub as (
  SELECT 
    t2.subject_id,
    COUNT(distinct channel_id) AS subs,
    COUNT(
      distinct CASE WHEN t1.day <= t2.date + interval '7 day' 
      AND t2.days >= 7 THEN channel_id ELSE NULL END
    ) AS d7_subs 
  FROM 
    warehouse.fact_viewer_activity_daily t1 
    JOIN metadata m ON t1.day >= m.start_date 
    AND t1.day <= m.end_date 
    JOIN exp t2 ON t1.device_id = t2.subject_id 
    AND t1.day >= t2.date 
  WHERE 
    viewer_is_subscribed_viewed_channel 
  GROUP BY 
    1
),
{extra_metrics_query}
pop AS (
  SELECT 
    exp.subject_id,
    exp.experiment_group,
    exp.date,
    exp.days,
    nvl(expo_vp.vp,0) AS vp,
    nvl(expo_vp.d0_vp,0) AS d0_vp,
    nvl(expo_vp.d7_vp,0) AS d7_vp,
    nvl(expo_fmp.fmp,0) AS fmp,
    nvl(expo_fmp.d0_fmp,0) AS d0_fmp,
    nvl(expo_fmp.d7_fmp,0) AS d7_fmp,
    nvl(expo_mw.mw,0) AS mw,
    nvl(expo_mw.mw_mobile,0) AS mw_mobile,
    nvl(expo_mw.d0_mw,0) AS d0_mw,
    nvl(expo_mw.d0_mw_mobile,0) AS d0_mw_mobile,
    nvl(expo_mw.d7_mw,0) AS d7_mw,
    nvl(expo_mw.d7_mw_mobile,0) AS d7_mw_mobile,
    nvl(expo_session.session,0) AS sessions,
    nvl(expo_pageview.pageview,0) AS pageview,
    nvl(expo_pageview.d0_pageview,0) AS d0_pageview,
    nvl(expo_pageview.d7_pageview,0) AS d7_pageview,
    {ret_pop}
    nvl(expo_pageview.logged_in) AS logged_in,
    nvl(expo_pageview.d7_logged_in,0) AS d7_logged_in,
    nvl(gf.game_follows,0) AS game_follows,
    nvl(gf.mobile_game_follows,0) AS mobile_game_follows,
    nvl(f.follows,0) as follows,
    nvl(sub.subs,0) as subs,
    nvl(sub.d7_subs) as d7_subs
    {extra_metrics_pop_select} 
  FROM 
    exp 
    LEFT JOIN expo_fmp ON expo_fmp.subject_id = exp.subject_id 
    LEFT JOIN expo_mw ON expo_mw.subject_id = exp.subject_id 
    LEFT JOIN expo_session ON expo_session.subject_id = exp.subject_id 
    LEFT JOIN expo_pageview ON expo_pageview.subject_id = exp.subject_id 
    LEFT JOIN expo_vp ON expo_vp.subject_id = exp.subject_id 
    LEFT JOIN game_follow gf ON gf.subject_id = exp.subject_id 
    LEFT JOIN follow f ON f.subject_id = exp.subject_id 
    LEFT JOIN sub ON sub.subject_id = exp.subject_id
    {extra_metrics_pop_join} 
  WHERE 
    experiment_group in (
      SELECT 
        experiment_group 
      FROM 
        detect_groups
    )
),
perm_test_cutoffs AS (
  SELECT 
    experiment_group,
    nvl(
      sum(pct) over(
        ORDER BY 
          experiment_group rows between unbounded preceding 
          AND 1 preceding
      ),
      0
    ) AS lower_cutoff,
    sum(pct) over(
      ORDER BY 
        experiment_group rows unbounded preceding
    ) AS upper_cutoff 
  FROM 
    detect_groups
),
iterations AS (
  SELECT 
    row_number() over (
      ORDER BY 
        "date"
    ) AS iteration 
  FROM 
    metadata."date" 
  LIMIT 
    2
),permuted_group AS (
  SELECT 
    pop.subject_id,
    i.iteration AS iteration,
    random() AS rnd 
  FROM 
    pop CROSS 
    JOIN iterations i
),
permuted_pop AS (
  SELECT 
    pop.subject_id,
    0 AS iteration,
    pop.experiment_group AS permuted_group 
  FROM 
    pop 
  UNION 
  SELECT 
    pop.subject_id,
    g.iteration AS iteration,
    c.experiment_group AS permuted_group 
  FROM 
    pop 
    JOIN permuted_group g ON pop.subject_id = g.subject_id 
    LEFT JOIN perm_test_cutoffs c ON g.rnd between lower_cutoff 
    AND upper_cutoff
) 
SELECT 
  pm.iteration,
  pm.permuted_group AS experiment_group,
  SUM(1) AS core_nsubjects,
  SUM(fmp) :: float / core_nsubjects AS avg_fmp,
  SUM(CASE WHEN fmp > 0 THEN 1 ELSE 0 END) :: float / core_nsubjects AS avg_fmp_conversion_pct,
  SUM(vp) :: float / core_nsubjects AS avg_vp,
  SUM(mw) :: float / core_nsubjects AS avg_mw,
  SUM(mw_mobile) :: float / core_nsubjects AS avg_mw_mobile,
  SUM(pageview) :: float / core_nsubjects AS avg_pageview,
  SUM(sessions) :: float / core_nsubjects AS avg_sessions,
  SUM(logged_in) :: float / core_nsubjects AS avg_logged_in,
  SUM(game_follows) :: float / core_nsubjects AS avg_game_follows,
  SUM(mobile_game_follows) :: float / core_nsubjects AS avg_mobile_game_follows,
  SUM(
    CASE WHEN mobile_game_follows > 0 THEN 1 ELSE 0 END
  ) :: float / core_nsubjects AS avg_mobile_game_follow_conversion_pct,
  SUM(follows) :: float / core_nsubjects AS avg_follows,
  SUM(subs) :: float / core_nsubjects AS avg_subs,
  SUM(d0_vp) :: float / core_nsubjects AS avg_d0_vp,
  SUM(d0_fmp) :: float / core_nsubjects AS avg_d0_fmp,
  SUM(CASE WHEN d0_fmp > 0 THEN 1 ELSE 0 END) :: float / core_nsubjects AS avg_d0_fmp_conversion_pct,
  SUM(d0_mw) :: float / core_nsubjects AS avg_d0_mw,
  SUM(d0_mw_mobile) :: float / core_nsubjects AS avg_d0_mw_mobile,
  SUM(d0_pageview) :: float / core_nsubjects AS avg_d0_pageview,
  {ret_agg}
  SUM(
    CASE WHEN pop.days >= 7 THEN 1 ELSE 0 END
  ) AS d7_nsubjects,
  CASE WHEN d7_nsubjects > 0 THEN SUM(d7_vp) :: FLOAT / d7_nsubjects ELSE 0 END AS avg_d7_vp,
  CASE WHEN d7_nsubjects > 0 THEN SUM(d7_fmp) :: FLOAT / d7_nsubjects ELSE 0 END AS avg_d7_fmp,
  CASE WHEN d7_nsubjects > 0 THEN SUM(CASE WHEN d7_fmp > 0 THEN 1 ELSE 0 END) :: FLOAT / d7_nsubjects ELSE 0 END AS avg_d7_fmp_conversion_pct,
  CASE WHEN d7_nsubjects > 0 THEN SUM(d7_mw) :: FLOAT / d7_nsubjects ELSE 0 END AS avg_d7_mw,
  CASE WHEN d7_nsubjects > 0 THEN SUM(d7_mw_mobile) :: FLOAT / d7_nsubjects ELSE 0 END AS avg_d7_mw_mobile,
  CASE WHEN d7_nsubjects > 0 THEN SUM(d7_logged_in) :: FLOAT / d7_nsubjects ELSE 0 END AS avg_d7_logged_in,
  CASE WHEN d7_nsubjects > 0 THEN SUM(d7_pageview) :: FLOAT / d7_nsubjects ELSE 0 END AS avg_d7_pageview,
  CASE WHEN d7_nsubjects > 0 THEN SUM(d7_subs) :: FLOAT / d7_nsubjects ELSE 0 END AS avg_d7_subs
  {extra_metrics_agg} 
FROM 
  pop 
  JOIN permuted_pop pm on pop.subject_id = pm.subject_id 
GROUP BY 
  1,
  2
