import time
from absl import app, flags, logging
from absl.flags import FLAGS
import psycopg2

from mgst_data.utils import date_validator
from mgst_data.config import DB_HOST, DB_USER, DB_PASSWORD
from mgst_data.tahoe_speedup import create_table, drop_table, copy_data

flags.DEFINE_string('start_date', None, 'Starting Date (2019-01-01)')
flags.DEFINE_string('end_date', None, 'Ending Date (2019-02-01)')
flags.DEFINE_string('drop_date', None, 'Ending Date (2019-02-01)')
flags.DEFINE_string('table_name', None, 'Table to copy')
flags.DEFINE_boolean('drop_table', False, 'Drop table')
flags.DEFINE_boolean('create_table', False, 'Create table')

flags.register_validator(
    'start_date', date_validator(), message='invalid date')
flags.register_validator('end_date', date_validator(), message='invalid date')
flags.register_validator('drop_date', date_validator(
    required=False), message='invalid date')
flags.mark_flag_as_required('table_name')


def main(_argv):
    start_time = time.time()
    logging.info("Processing %s from %s to %s", FLAGS.table_name,
                 FLAGS.start_date, FLAGS.end_date)

    con = psycopg2.connect(dbname='product', host=DB_HOST,
                           port='5439', user=DB_USER, password=DB_PASSWORD)

    cursor = con.cursor()

    if FLAGS.drop_table:
        sql = drop_table(FLAGS.table_name)
        cursor.execute(sql)
        con.commit()

    if FLAGS.create_table or FLAGS.drop_table:
        sql = create_table(FLAGS.table_name)
        cursor.execute(sql)
        cursor.execute(f'GRANT ALL ON mgst."{FLAGS.table_name}" TO mode_ro;')
        con.commit()

    cursor.execute(f"""
        DELETE FROM mgst."{FLAGS.table_name}"
        WHERE date BETWEEN %s AND %s
    """, [FLAGS.start_date, FLAGS.end_date])

    sql = copy_data(FLAGS.table_name)
    cursor.execute(sql, {
        'start_date': FLAGS.start_date,
        'end_date': FLAGS.end_date
    })
    con.commit()

    logging.info("Processed %d rows", cursor.rowcount)

    if FLAGS.drop_date:
        cursor.execute(f"""
            DELETE FROM mgst."{FLAGS.table_name}"
            WHERE date < %s
        """, [FLAGS.drop_date])
        logging.info("Dropped %d rows", cursor.rowcount)
        con.commit()

    logging.info("Done in %ds", time.time() - start_time)


if __name__ == '__main__':
    app.run(main)
