resource "aws_security_group" "eks-jumpbox" {
  name_prefix = "${var.name}-eks-jumpbox"
  vpc_id      = data.aws_vpc.tahoe-vpc.id

  ingress {
    from_port = 0
    to_port   = 0
    protocol  = "-1"

    security_groups = [data.aws_security_group.teleport-jumpbox.id]
  }
}

module "eks-cluster" {
  # source          = "terraform-aws-modules/eks/aws"
  source          = "git@github.com:terraform-aws-modules/terraform-aws-eks.git?ref=v11.1.0"
  cluster_name    = "${var.name}-eks"
  cluster_version = "1.15"

  write_kubeconfig      = false
  vpc_id                = data.aws_vpc.tahoe-vpc.id
  subnets               = concat(data.aws_subnet.tahoe-subnet-public.*.id, data.aws_subnet.tahoe-subnet-private.*.id)

  worker_groups_launch_template = [{
    name                   = "worker-group-1"
    instance_type          = "r5.xlarge"
    additional_userdata   = var.install-ssm-agent-cmd
    asg_desired_capacity   = 1
    asg_min_size           = 1
    asg_max_size           = 10
    autoscaling_enabled    = true
    key_name               = aws_key_pair.ssh.key_name
    subnets                = data.aws_subnet.tahoe-subnet-private.*.id
    asg_recreate_on_change = true
    },
    {
      name                 = "jupyter-group-1"
      instance_type        = "t3.small"
      additional_userdata   = var.install-ssm-agent-cmd
      asg_desired_capacity = 1
      asg_min_size         = 1
      asg_max_size         = 1
      kubelet_extra_args = join(" ", [
        format("--node-labels=%s",
          join(",", [
            "hub.jupyter.org/node-purpose=core"
        ]))
      ])
      key_name               = aws_key_pair.ssh.key_name
      subnets                = [data.aws_subnet.tahoe-subnet-private[0].id]
      asg_recreate_on_change = true
    },
    {
      name                 = "jupyter-group-2"
      instance_type        = "m5.large"
      # install ssm agent
      additional_userdata   = var.install-ssm-agent-cmd
      asg_desired_capacity = 0
      asg_min_size         = 0
      asg_max_size         = 10
      autoscaling_enabled  = true
      kubelet_extra_args = join(" ", [
        format("--node-labels=%s",
          join(",", [
            "hub.jupyter.org/node-purpose=user"
        ])),
        "--register-with-taints=hub.jupyter.org/dedicated=user:NoSchedule"
      ])
      key_name               = aws_key_pair.ssh.key_name
      subnets                = [data.aws_subnet.tahoe-subnet-private[0].id]
      asg_recreate_on_change = true
    },
    {
      name                   = "compute-group-1"
      instance_type          = "m5.2xlarge"
      # install ssm agent
      additional_userdata   = var.install-ssm-agent-cmd
      asg_desired_capacity   = 0
      asg_min_size           = 0
      asg_max_size           = 100
      autoscaling_enabled    = true
      key_name               = aws_key_pair.ssh.key_name
      subnets                = [data.aws_subnet.tahoe-subnet-private[0].id]
      asg_recreate_on_change = true
    },
  ]

  worker_additional_security_group_ids = [aws_security_group.eks-jumpbox.id]
}

locals {
  # TODO: 
  #   see https://github.com/terraform-providers/terraform-provider-aws/issues/10104
  #   and https://github.com/terraform-providers/terraform-provider-tls/issues/52
  eks-oidc-thumbprint = "9e99a48a9960b14926bb7f3b02e22da2b0ab7280"
}

resource "aws_iam_openid_connect_provider" "eks-oidc" {
  client_id_list  = ["sts.amazonaws.com"]
  thumbprint_list = [local.eks-oidc-thumbprint]
  url             = module.eks-cluster.cluster_oidc_issuer_url
}
