resource "aws_iam_policy" "dns-manager" {
  name = "${var.name}-dns-manager"

  policy = <<EOF
{
"Version": "2012-10-17",
"Statement": [
  {
    "Effect": "Allow",
    "Action": [
      "route53:ChangeResourceRecordSets"
    ],
    "Resource": [
      "arn:aws:route53:::hostedzone/*"
    ]
  },
  {
    "Effect": "Allow",
    "Action": [
      "route53:ListHostedZones",
      "route53:ListResourceRecordSets"
    ],
    "Resource": [
      "*"
    ]
  }
]
}
EOF
}

module "iam-external-dns" {
  source = "./eks-iam"

  name           = "${var.name}-eks-external-dns"
  serviceaccount = "default:external-dns"
  policy_arns    = [aws_iam_policy.dns-manager.arn]

  oidc_arn = aws_iam_openid_connect_provider.eks-oidc.arn
  oidc_url = aws_iam_openid_connect_provider.eks-oidc.url
}

resource "aws_iam_policy" "ssm-reader" {
  name = "${var.name}-ssm-reader"

  policy = <<EOF
{
"Version": "2012-10-17",
"Statement": [
  {
    "Effect": "Allow",
    "Action": [
      "ssm:GetParameter"
    ],
    "Resource": [
      "arn:aws:ssm:${var.region}:${data.aws_caller_identity.current.account_id}:*"
    ]
  }
]
}
EOF
}

module "iam-mgst-data" {
  source = "./eks-iam"

  name           = "${var.name}-eks-mgst-data"
  serviceaccount = "mgst-data:mgst-data"
  policy_arns    = [aws_iam_policy.ssm-reader.arn, "arn:aws:iam::aws:policy/AmazonS3FullAccess"]

  oidc_arn = aws_iam_openid_connect_provider.eks-oidc.arn
  oidc_url = aws_iam_openid_connect_provider.eks-oidc.url
}

module "iam-jupyter" {
  source = "./eks-iam"

  name           = "${var.name}-eks-jupyter"
  serviceaccount = "jupyterhub:jupyter"
  policy_arns    = [aws_iam_policy.ssm-reader.arn]

  oidc_arn = aws_iam_openid_connect_provider.eks-oidc.arn
  oidc_url = aws_iam_openid_connect_provider.eks-oidc.url
}

module "iam-fluent-bit-cloudwatch" {
  source = "./eks-iam"

  name           = "${var.name}-eks-fluent-bit-cloudwatch"
  serviceaccount = "default:fluent-bit-cloudwatch"
  policy_arns    = ["arn:aws:iam::aws:policy/CloudWatchAgentServerPolicy"]

  oidc_arn = aws_iam_openid_connect_provider.eks-oidc.arn
  oidc_url = aws_iam_openid_connect_provider.eks-oidc.url
}

resource "aws_iam_policy" "cluster-autoscaler" {
  name = "${var.name}-cluster-autoscaler"

  policy = <<EOF
{
"Version": "2012-10-17",
"Statement": [
  {
      "Effect": "Allow",
      "Action": [
          "autoscaling:DescribeAutoScalingGroups",
          "autoscaling:DescribeAutoScalingInstances",
          "autoscaling:DescribeTags",
          "autoscaling:DescribeLaunchConfigurations",
          "autoscaling:SetDesiredCapacity",
          "autoscaling:TerminateInstanceInAutoScalingGroup",
          "ec2:DescribeLaunchTemplateVersions"
      ],
      "Resource": "*"
  }
]
}
EOF
}

module "iam-cluster-autoscaler" {
  source = "./eks-iam"

  name           = "${var.name}-cluster-autoscaler"
  serviceaccount = "default:cluster-autoscaler"
  policy_arns    = [aws_iam_policy.cluster-autoscaler.arn]

  oidc_arn = aws_iam_openid_connect_provider.eks-oidc.arn
  oidc_url = aws_iam_openid_connect_provider.eks-oidc.url
}

# module "iam-spark-worker" {
#   source = "./eks-iam"

#   name           = "${var.name}-spark-worker"
#   serviceaccount = "spark:spark-worker"
#   policy_arns    = [data.terraform_remote_state.tahoe-admin.outputs.redshift_iam_role]

#   oidc_arn = aws_iam_openid_connect_provider.eks-oidc.arn
#   oidc_url = aws_iam_openid_connect_provider.eks-oidc.url
# }

# Grant IAM privileges for SSM Agent.
resource "aws_iam_role_policy_attachment" "ssm_policy" {
  role = module.eks-cluster.worker_iam_role_name
  policy_arn = "arn:aws:iam::aws:policy/AmazonSSMManagedInstanceCore"
}
