import traceback
import sys
import random
import time

import gi
gi.require_version('Gst', '1.0')

from gi.repository import Gst, GObject, GLib, Gio
GObject.threads_init()
Gst.init(None)
Gio.networking_init()

TEST_GPU = False
TEST_COUNT = 1
TEST_SLEEP_DELAY = 1
TEST_ENCODE_PNG = False
TEST_FRAMERATE = 1



def link_many(*args):
    if not len(args):
        return
    for index, element in enumerate(args[:-1]):
        next_element = args[index+1]
        if not Gst.Element.link(element, next_element):
            raise TypeError('failed to link {}, {}'.format(element, next_element))


def make_queue(queue_name="", **kwargs):
    props = {k.replace("_", "-"): v for k, v in kwargs.items()}
    if queue_name:
        queue_name += "_"
    name = "%s_queue" % (queue_name)
    queue = Gst.ElementFactory.make("queue", name)
    queue.set_property("max-size-bytes", props.pop("max-size-bytes", 0))
    queue.set_property("max-size-buffers", props.pop("max-size-buffers", 0))
    queue.set_property("max-size-time", props.pop("max-size-time", 0))
    queue.set_property("silent", props.pop("silent", True))
    for k, v in props.items():
        queue.set_property(k, v)
    return queue

def make(element_name, name=None, **kwargs):
    props = {k.replace("_", "-"): v for k, v in kwargs.items()}
    element = Gst.ElementFactory.make(element_name, name)
    if not element:
        raise TypeError('Failed to create element: {}, {} {}'.format(element_name, name, props))
    for k, v in props.items():
        element.set_property(k, v)
    return element

def make_caps(caps_str):
    return make("capsfilter", caps=Gst.Caps.from_string(caps_str))



class Pipeline:
    def __init__(self, i):
        self.id = str(i)
        self.pipeline = None
        self.lt = 0
        self.klt = 0

    def fakesink_handoff(self, _, buf, pad):
        caps = pad.get_current_caps().to_string()
        now = self.pipeline.get_pipeline_clock().get_time()
        if self.lt == 0:
            print(caps)
        self.lt = now

    def parser_buffer_probe_cb(self, pad, info):
        buf = info.get_buffer()
        if buf.has_flags(Gst.BufferFlags.DELTA_UNIT): # Not keyframe
            return Gst.PadProbeReturn.DROP

        now = self.pipeline.get_pipeline_clock().get_time()
        print("pipeline id:", self.id, "time:", (now - self.klt) / 1e6, "KEYFRAME FOUND", buf)
        self.klt = now

        return Gst.PadProbeReturn.PASS

    def on_tsdemux_cb(self, element, pad):
        caps = pad.get_current_caps().to_string()
        print("on_ts_demux caps:", caps)

        if not "video" in caps:
            return

        parser = make("h264parse")
        parser.get_static_pad("src").add_probe(Gst.PadProbeType.BUFFER, self.parser_buffer_probe_cb)

        decode_queue_pre = make_queue("decoder-pre", max_size_buffers=0, leaky=2)
        decoder = make("nvdec") if TEST_GPU else make("avdec_h264")
        decode_queue_post = make_queue("decoder-post", max_size_buffers=0, leaky=2)
        sink = make("fakesink", signal_handoffs=True)
        sink.connect("handoff", self.fakesink_handoff)

        elements = [parser, decode_queue_pre, decoder, decode_queue_post, sink]

        for s in elements:
            self.pipeline.add(s)

        pad.link(elements[0].get_static_pad('sink'))
        link_many(*elements)

        for e in elements:
            e.set_state(Gst.State.PLAYING)


    def on_hlsdemux_cb(self, element, pad):
        caps = pad.get_current_caps().to_string()
        print("on_hls_demux caps:", caps)

        if not "mpegts" in caps:
            return

        ts_demuxer = make("tsdemux")
        self.pipeline.add(ts_demuxer)
        ts_demuxer.connect("pad-added", self.on_tsdemux_cb)
        pad.link(ts_demuxer.get_static_pad("sink"))
        ts_demuxer.set_state(Gst.State.PLAYING)

    def start(self):
        self.pipeline = Gst.Pipeline.new("mirage_" + str(self.id))

        # Fortnite VOD
        uri="http://184.72.239.149/vod/smil:BigBuckBunny.smil/playlist.m3u8"
        uri="https://content.jwplatform.com/manifests/vM7nH0Kl.m3u8"
        uri="https://vod-metro.twitch.tv/3a15a6fc728a3e38f359_ninja_34825785696_1245279696/chunked/index-dvr.m3u8"
        uri="https://vod-metro.twitch.tv/77ba829436b44bb6dd68_lord_kebun_34957558560_1253517113/chunked/index-dvr.m3u8"
        uri="https://usher.ttvnw.net/api/channel/hls/moonmoon_ow.m3u8?allow_source=true&baking_bread=true&baking_brownies=true&baking_brownies_timeout=1050&fast_bread=true&p=3144012&player_backend=mediaplayer&playlist_include_framerate=true&reassignments_supported=true&sig=ce1a7883da38175b98a39aef36322a5d56f88918&token=%7B%22adblock%22%3Atrue%2C%22authorization%22%3A%7B%22forbidden%22%3Afalse%2C%22reason%22%3A%22%22%7D%2C%22blackout_enabled%22%3Afalse%2C%22channel%22%3A%22moonmoon_ow%22%2C%22channel_id%22%3A121059319%2C%22chansub%22%3A%7B%22restricted_bitrates%22%3A%5B%5D%2C%22view_until%22%3A1924905600%7D%2C%22ci_gb%22%3Afalse%2C%22geoblock_reason%22%3A%22%22%2C%22device_id%22%3A%22103870b6cfb5d7b8%22%2C%22expires%22%3A1563591013%2C%22game%22%3A%22Super%20Mario%20Maker%202%22%2C%22hide_ads%22%3Afalse%2C%22https_required%22%3Atrue%2C%22mature%22%3Afalse%2C%22partner%22%3Afalse%2C%22platform%22%3A%22web%22%2C%22player_type%22%3A%22site%22%2C%22private%22%3A%7B%22allowed_to_view%22%3Atrue%7D%2C%22privileged%22%3Afalse%2C%22server_ads%22%3Atrue%2C%22show_ads%22%3Atrue%2C%22subscriber%22%3Afalse%2C%22turbo%22%3Afalse%2C%22user_id%22%3A139658194%2C%22user_ip%22%3A%22204.246.162.34%22%2C%22version%22%3A2%7D&supported_codecs=vp09%2Cavc1&cdm=wv"
        uri="https://usher.ttvnw.net/api/channel/hls/moonmoon_ow.m3u8?allow_source=true&baking_bread=true&baking_brownies=true&baking_brownies_timeout=1050&fast_bread=true&p=7490240&player_backend=mediaplayer&playlist_include_framerate=true&reassignments_supported=true&sig=d6334c7c1f803065633973c495db02955b33b0e4&token=%7B%22adblock%22%3Atrue%2C%22authorization%22%3A%7B%22forbidden%22%3Afalse%2C%22reason%22%3A%22%22%7D%2C%22blackout_enabled%22%3Afalse%2C%22channel%22%3A%22moonmoon_ow%22%2C%22channel_id%22%3A121059319%2C%22chansub%22%3A%7B%22restricted_bitrates%22%3A%5B%5D%2C%22view_until%22%3A1924905600%7D%2C%22ci_gb%22%3Afalse%2C%22geoblock_reason%22%3A%22%22%2C%22device_id%22%3A%22103870b6cfb5d7b8%22%2C%22expires%22%3A1563593644%2C%22game%22%3A%22Super%20Mario%20Maker%202%22%2C%22hide_ads%22%3Afalse%2C%22https_required%22%3Atrue%2C%22mature%22%3Afalse%2C%22partner%22%3Afalse%2C%22platform%22%3A%22web%22%2C%22player_type%22%3A%22site%22%2C%22private%22%3A%7B%22allowed_to_view%22%3Atrue%7D%2C%22privileged%22%3Afalse%2C%22server_ads%22%3Atrue%2C%22show_ads%22%3Atrue%2C%22subscriber%22%3Afalse%2C%22turbo%22%3Afalse%2C%22user_id%22%3A139658194%2C%22user_ip%22%3A%22204.246.162.34%22%2C%22version%22%3A2%7D&supported_codecs=vp09%2Cavc1&cdm=wv"
        # gst-launch-1.0 souphttpsrc ! hlsdemux ! tsdemux name=mux mux. ! h264parse ! identity drop-buffer-flags=0x00002000 ! avdec_h264 ! videoconvert ! autovideosink

        src = make("souphttpsrc", is_live=True, location=uri)
        demux = make("hlsdemux")
        demux.connect("pad-added", self.on_hlsdemux_cb)

        elements = [src, demux]

        for s in elements:
            self.pipeline.add(s)

        link_many(*elements)

        self.pipeline.set_state(Gst.State.PLAYING)
        print(self.id, "pipeline set to playing")




def main():
    print("Starting", TEST_COUNT, "pipelines. GPU:", TEST_GPU)

    for i in range(TEST_COUNT):
        p = Pipeline(i)
        p.start()
        time.sleep(TEST_SLEEP_DELAY)

    print("all streams running")


if __name__ == '__main__':
    try:
        main()

        loop = GLib.MainLoop()
        loop.run()
    except KeyboardInterrupt:
        print("stopping")
    except Exception:
        traceback.print_exc(file=sys.stdout)
        print("sh*t blows")
        sys.exit(1)


