const config = require('../config');
const logger = require('../api/logger');
const request = require('request-promise');
const queue = require('async/queue');

const WSClient = require('./ws_client');

const sleep = m => new Promise(r => setTimeout(r, m));
const Promise = require('bluebird');

logger.init({ name: 'winnerbot' });


function url2idkey(url) {
  let id_key = url.replace(/^\/+|\/+$/g,"");
  id_key = id_key.replace(/\/+/g,"_") + "_id";
  return id_key;
}

function getId(entity) {
  let id_key = url2idkey(entity.url);
  return entity[id_key];
}


// TODO expire old id data

class BaseBot {
  constructor() {
    this.intervals = {};
    this.last = {};
    this.leagues = {};
    this.tournaments = {};
    this.checkins = new Set();
    this.queue = queue(async (task, callback) => {
      try {
        if (task.updated_dttm && task.url) {
          let updated = new Date(task.updated_dttm);
          let id = getId(task);
          let last = this.last[id] || new Date(0);
          // logger.debug(last, updated);
          if (updated > last) {
            await this.doFSM(task);
          } else {
            logger.warning(this.user && this.user.username || "-", 
              "dropping late message", task.url, id, task.updated_dttm);
          }
          this.last[id]  = updated;
        } else {
          logger.warning(this.user && this.user.username || "-",
            "no updated_dttm", task.url);
          await this.doFSM(task);
        }
      } catch (err) {
        logger.error(this.user && this.user.username || "-",
          'Message Processing Error', err, JSON.stringify(task, null, 2));
      }
      callback();
    }, 1);

    this.goIdle();
  }

  async login(user) {
    const userRes = await request(config.aws_api_url + '/user/login', {
      body: {
        username: user.username,
        password: user.password
      },
      method: 'POST',
      json: true
    });

    this.user = Object.assign(user, userRes.result[0]);
  }

  async setState(state, data, expire, once) {
    clearInterval(this.intervals[state]);
    if (! expire) {
      expire = 90;
    }
    const setStateFunc = () => {
      const msg = {
        method: 'PUT',
        url: '/user/state/' + state,
        expire
      };
      if (data) {
        msg.data = data;
      }
      return this.send(msg);
    };
    if (! once) {
      this.intervals[state] = setInterval(setStateFunc, 60 * 1000);
    }
    return setStateFunc();
  }

  async send(msg) {
    return new Promise((resolve, reject) => {
      let onResult = result => {
        if (result.code === 200) {
          if (msg.method !== "GET" && result.result && result.result.length === 1) {
            resolve(result.result[0]);
          } else {
            resolve(result);
          }
        } else {
          logger.error('rejecting', result);
          reject(result);
        }
      };
      logger.debug(this.user && this.user.username || "-", "send", msg);
        
      this.ws.wsSend(msg, onResult);
    });
  }

  async goIdle() {
    this.state = 'idle';
    this.match = {};
    this.set = {};
  }

  async logResponseError(msg) {
    if (msg.code !== 200) {
      logger.error(msg);
    }
  }

  async subscribeMatch(match_id) {
    let msg = {
      method: 'POST',
      url: '/subscribe/match',
      match_id: match_id
    };
    // logger.debug("subscribe", msg);
    let match = await this.send(msg);
    logger.debug('subscribe result', match.match_id, this.getMySet(match).state);
    this.queue.push(match);
  }

  async subscribeLeague(league_id) {
    let msg = {
      method: 'POST',
      url: '/subscribe/league',
      league_id: league_id
    };
    // logger.debug("subscribe", msg);
    let league = await this.send(msg);
    logger.debug('subscribe result', league.league_id);
    this.queue.push(league);
  }
  async subscribeTournament(tournament_id) {
    let msg = {
      method: 'POST',
      url: '/subscribe/tournament',
      tournament_id: tournament_id
    };
    // logger.debug("subscribe", msg);
    let tournament = await this.send(msg);
    this.queue.push(tournament);
  }
  async checkin(league) {

    let msg = {
      method: 'GET',
      url: '/tournament',
      league_id: league.league_id,
      state: 'created',
      deleted_dttm: 'null',
      count: 100
    };
    let tournamentsResponse = await this.send(msg);

    for (const tournament of tournamentsResponse.result) {
      let tournament_id = tournament.tournament_id;

      let team_id = league.teams[0] && league.teams[0].team_id;
      if (team_id == null || this.checkins.has(team_id)) {
        continue;
      }

      this.tournaments[tournament_id] = tournament;
      this.subscribeTournament(tournament_id);

      msg = {
        method: 'POST',
        url: '/tournament/team/checkin',
        tournament_id,
        team_id
      };
      await this.send(msg);
      this.checkins.add(team_id);
      logger.info(this.user.username, 'checked into tournament', tournament.tournament_id);
    }
  }

  async simulateLobbyReady() {
    await Promise.delay(Math.random() * 30000);
    await this.setState('walle');
    await this.setState('bran');
    await this.setState('fortwatch');
    let data = {
      user_id: this.user.user_id,
      match_id: this.match.match_id,
    };
    await this.setState('ready', data, 600, true);
  }

  async startGame() {
    let msg = {
      method: 'POST',
      url: '/user/game',
      started_dttm: '$NOW'
    };
    let game = await this.send(msg);
    this.user_game_id = game.user_game_id;
    logger.info(this.match.match_id, 'started game round', this.round);
  }

  async finishGame() {
    let msg = {
      method: 'PUT',
      url: '/user/game',
      ended_dttm: '$NOW'
    };
    if (this.user_game_id) {
      msg.user_game_id = this.user_game_id;
      this.user_game_id = null;
    }
    this.ws.wsSend(msg, this.logResponseError);
    logger.info(this.match.match_id, 'game ended round', this.round);
  }

  async start() {
    if (!this.user) {
      logger.error('Bot not logged in');
      return;
    }
    logger.info('starting bot', this.user);
    this.ws = new WSClient(this.user, async user => {
      logger.debug('user subscribe', user);
      this.ws.setMessageListener(msg => this.onMessage(msg));
      await this.setState('pc');
      this.queue.unshift(user.result[0]);
    });
  }

  async playRound() {
    try {
      await this.startGame();
      await sleep(3000);
      for (let i = 0; i < Math.floor(Math.random() * 5); i += 1) {
        await this.kill();
        await sleep(3000);
      }
      await sleep(3000);
      if (Math.random() > 0.8) {
        await this.victory();
      }
      await sleep(20000);
      await this.finishGame();
    } catch (err) {
      logger.error(err);
    }
  }

  async kill() {
    let options = {
      method: 'POST',
      url: `${config.aws_api_url}/user/kill`,
      headers: {
        'cache-control': 'no-cache',
        'Content-Type': 'application/json',
        'X-User-Id': this.user.user_id,
        'X-API-Key': config.SERVICE_API_KEY
      },
      body: {
        method: 'POST',
        url: '/user/kill',
        game: 'Fortnite',
        image_url: 'https://a.imgdropt-dev.com/image/71231950-8dc1-43be-8a23-12b483d3ab18',
        verified: false,
        yes_value: 100,
        no_value: 0
      },
      json: true
    };

    await request(options);
    logger.info(this.match.match_id, 'kill');
  }
  async victory() {
    let options = {
      method: 'POST',
      url: `${config.aws_api_url}/user/victory`,
      headers: {
        'cache-control': 'no-cache',
        'Content-Type': 'application/json',
        'X-User-Id': this.user.user_id,
        'X-API-Key': config.SERVICE_API_KEY
      },
      body: {
        method: 'POST',
        url: '/user/victory',
        game: 'Fortnite',
        image_url: 'https://a.imgdropt-dev.com/image/71231950-8dc1-43be-8a23-12b483d3ab18',
        verified: false,
        yes_value: 100,
        no_value: 0
      },
      json: true
    };

    await request(options);
    logger.info(this.match.match_id, 'victory');
  }

  getMySet(match) {
    const set = match.sets.find(s => s.team.users.some(u => u.user_id === this.user.user_id));
    if (!set) {
      logger.error('bot not in match', this.user.user_name, this.user.user_id, JSON.stringify(match, null, 2));
      throw new Error('Bot not in Match??');
    }
    return set;
  }

  async doFSM(entity) {
    let oldState = this.state;
    if (entity.url === '/match') {
      this.match = entity;
      this.set = this.getMySet(entity);
    }
    logger.info('doFSM', this.state, entity.url, this.set && this.set.state);
    let match_id = this.match.match_id;

    if (entity.url === '/match' && entity.deleted_dttm) {
      await this.goIdle();
    } else if (entity.url === '/user' && entity.active_match_id) {
      match_id = entity.active_match_id;
      logger.info('Enter new match', match_id);
      await this.subscribeMatch(entity.active_match_id);
    } else if (this.state === 'idle') {
      if (entity.url === '/match') {
        if (this.set.state === 'waiting') {
          this.state = 'waiting';
          logger.info('Getting ready for match', this.match.match_id);
          await this.simulateLobbyReady();
        } else if (this.set.state === 'started') {
          this.state = 'waiting';
          this.queue.push(entity);
        }
      }
    } else if (this.state === 'waiting') {
      if (entity.url === '/match') {
        if (this.set.state === 'started') {
          this.state = 'game';
          this.round = this.set.round;
          if (this.round <= this.match.game_cnt) {
            logger.info('Playing round', this.round);
            await this.playRound(entity);
          }
        } else if (this.set.state === 'waiting') {
          let mystate = this.match.data.state[this.user.user_id] || {};
          if (! mystate.ready) {
            logger.info('Getting ready for match (again)', this.match.match_id);
            await this.simulateLobbyReady();
          }
        }
      }
    } else if (this.state === 'game') {
      if (this.set.state === 'started') {
        if (this.round < this.set.round && this.set.round <= this.match.game_cnt) {
          this.round = this.set.round;
          logger.info('Playing round', this.round);
          await this.playRound(entity);
        }
      } else if (this.set.state === 'ended') {
        logger.info(entity.match_id, this.set.set_id, 'set ended');
        this.goIdle();
      }
    } else {
      logger.info('unknown handled state messate', entity);
    }

    if (this.state !== oldState) {
      if (this.match.match_id) {
        match_id = this.match.match_id;
      }
      logger.info(match_id, 'state transition', oldState, '->', this.state);
    }
  }

  async onMessage(msg) {
    if (!this.routes.has(msg.url)) {
      // logger.debug("dropped", msg.url);
      return;
    }
    if (msg.result) {
      this.queue.push(msg.result[0]);
    } else {
      logger.warn('onMessage no result', msg);
    }
  }
}

module.exports = BaseBot;
