const request = require('request-promise');
const Promise = require('bluebird');

const config = require('../config');
const logger = require('../api/logger');
const readline = require('readline').createInterface({
  input: process.stdin,
  output: process.stdout
});

const BaseBot = require('./base_bot');

const ADMIN_HEADERS = {
  'X-Api-Key': config.SERVICE_API_KEY
};

const TOURNAMENT_TEMPLATE = {
  name: `Rivals`,
  config: {
  },
  points_needed: 0
};


async function doAdminRequest(passedOptions) {
  const headers = Object.assign({}, passedOptions.headers || {}, ADMIN_HEADERS);
  const options = Object.assign({}, passedOptions, { headers, json: true });
  logger.info(`${options.method} ${options.url}`, options.body || {});
  return request(options);
}

function getline(str) {
  return new Promise(resolve => {
    readline.question(str + ' ', answer => {
      resolve(answer);
    });
  });
}

async function startNewTourney(league_id, options) {

  let {tournament_type, delay}  = options;

  const response = await doAdminRequest({
    url: `${config.aws_api_url}/tournament?league_id=${league_id}&state=started&count=100`,
    headers: ADMIN_HEADERS,
    method: 'GET'
  });

  response.result.map(async tournament => {
    await doAdminRequest({ method: 'PUT', body: { tournament_id: tournament.tournament_id, state: 'ended', end_dttm: "$NOW" }, url: `${config.aws_api_url}/tournament` });
  });

  let tournament = Object.assign({}, TOURNAMENT_TEMPLATE, {
    league_id,
    start_dttm: Math.floor((Date.now() + delay) / 60000) * 60000,
    type: tournament_type
  });

  await doAdminRequest({ url: `${config.aws_api_url}/league`, method: 'PUT', body: { league_id, active_tournament_id: null } });
  let tournamentResult = await doAdminRequest({ url: `${config.aws_api_url}/tournament`, method: 'POST', body: tournament});
  logger.info("created tournament", tournamentResult.result[0].tournament_id);

}

class DuoBot extends BaseBot {
  constructor(i) {
    super();
    this.win = i;
  }

  async login(user) {
    await super.login(user);
  }

  async playRound() {
    try {
      await Promise.delay(1000);
      await this.startGame();
      if (this.win) {
        for (let i = 0; i < Math.floor(this.win/5); i += 1) {
          await this.kill();
        }
        await this.victory();
      }
      await Promise.delay(1000);
      await this.finishGame();
    } catch (err) {
      logger.error(err);
    }
  }

  async doFSM(entity) {
    logger.debug(this.user.username, JSON.stringify(entity, null, 2));
    let oldState = this.state;
    let old_match_id = this.match && this.match.match_id;
    if (entity.url === '/match') {
      this.match = entity;
      this.set = this.getMySet(entity);
    }
    logger.info(this.user.username, 'doFSM', this.state, entity.url, this.set && this.set.state);
    let match_id = this.match.match_id;

    if (entity.url === '/league') {
      await this.checkin(entity);
    } else if (entity.url === '/match' && entity.deleted_dttm) {
      await this.goIdle();
    } else if (entity.url === '/user') {
      if (entity.active_match_id) {
        match_id = entity.active_match_id;
        if (old_match_id !== match_id) {
          this.state = 'idle';
          logger.info(this.user.username, 'Enter new match', match_id);
        }
        await this.subscribeMatch(entity.active_match_id);
      }
      for (const league of entity.leagues) {
        await this.subscribeLeague(league.league_id);
      }
    } else if (this.state === 'idle') {
      if (entity.url === '/match') {
        if (this.set.state === 'waiting') {
          this.state = 'waiting';
          logger.info(this.user.username, 'Getting ready for match', this.match.match_id);
          await this.simulateLobbyReady();
        } else if (this.set.state === 'started') {
          this.state = 'game';
          this.round = this.set.round;
          if (this.round <= this.match.game_cnt) {
            logger.info(this.user.username, 'Playing round', this.round);
            await this.playRound(entity);
          }
        }
      }
    } else if (this.state === 'waiting') {
      if (entity.url === '/match') {
        if (this.set.state === 'started') {
          this.state = 'game';
          this.round = this.set.round;
          if (this.round <= this.match.game_cnt) {
            logger.info(this.user.username, 'Playing round', this.round);
            await this.playRound(entity);
          }
        }
        if (this.set.state === 'waiting') {
          logger.info(this.user.username, 'Getting ready for match (again)', this.match.match_id);
          await this.simulateLobbyReady();
        }
      }
    } else if (this.state === 'game') {
      if (this.set.state === 'started') {
        if (this.round < this.set.round && this.set.round <= this.match.game_cnt) {
          this.round = this.set.round;
          logger.info(this.user.username, 'Playing round', this.round);
          await this.playRound(entity);
        }
      } else if (this.set.state === 'ended') {
        logger.info(this.user.username, entity.match_id, this.set.set_id, 'set ended');
        this.goIdle();
      }
    } else {
      logger.info(this.user.username, 'unknown handled state messate', entity);
    }

    if (this.state !== oldState) {
      if (this.match.match_id) {
        match_id = this.match.match_id;
      }
      logger.info(this.user.username, match_id, 'state transition', oldState, '->', this.state);
    }
  }
}

async function run(league_id, bot_count, logins, options) {
  await startNewTourney(league_id, options);
  await getline('start bots ? (enter)');
  const bots = [];
  for (let i = 0; i < bot_count; i++) {
    try {
      const bot = new DuoBot(i);
      logger.info('created bot', i);
      await bot.login(logins[i]);
      bots.push(bot.start());
      
    } catch(err) {
      logger.error(`Bot #${i} error`, err);
      logger.error('Unable to start bot', i);
    }
  }
  await Promise.all(bots);
}

module.exports = run;
