const request = require('request-promise');
const Promise = require('bluebird');
const stream = require('stream');

let bracketBot = null;
let stormBot = null;
const logger = require('../api/logger');
const config = require('../api/config');
const Broadcast = require('../api/broadcast');
const Dispatcher = require('../api/dispatcher');
const Analytics = require('../classes/analytics');

let UserController;
let LeagueModel;
let InviteModel;
let UserTeamModel;
let InviteController;

const argv = require('yargs')
  .usage(`Usage: node $0 [-t tournament_type] team_count`)
  .default('t', "storm")
  .default('team-size', 1)
  .describe('t', 'tournament_type')
  .default('delay', 90)
  .describe('delay', 'delay in seconds (min 90)')
  .choices('t', ["storm", "double_elim", "single_elim"])
  .describe('team-size', 'team size')
  .demandCommand(1)
  .argv;

const tournament_type = argv.t;
const team_size = argv['team-size'];
const delay = Math.max(90, parseInt(argv['delay'])) * 1000;
const team_count = parseInt(argv._[0]);
const player_count = team_size * team_count;
let PLAYER_PASSWORD = null;
const CAMPAIGN_NAME = 'tourney_league_spawner';
// let LEAGUE_NAME = `Bot Battle - us against them`;
let LEAGUE_NAME = `Florian Single Elim - ${team_size}`;
const TEAM_INVITE_CODES = [];


const readline = require('readline').createInterface({
  input: process.stdin,
  output: process.stdout
});

function getLine(str) {
  return new Promise(resolve => {
    readline.question(str + ' ', answer => {
      resolve(answer);
    });
  });
}

function getPass(str) {
  return new Promise(resolve => {
    readline.question(str + ' ', answer => {
      process.stdout.clearLine(0);
      readline.history = readline.history.slice(1);
      readline.close();
      resolve(answer);
    });
  });
}

async function doAdminRequest(passedOptions) {
  const headers = Object.assign({}, passedOptions.headers || {},
    { 'X-Api-Key': config.SERVICE_API_KEY });
  const options = Object.assign({}, passedOptions, { headers, json: true });
  logger.info(`${options.method} ${options.url}`, options.body || {});
  return request(options);
}

async function getOrCreateLeague(name) {
  return LeagueModel.findAll2({ where: { name } })
    .then(async league => {
      if (league && league.length) {
        return league[0];
      }
      const response = await doAdminRequest({ url: `${config.aws_api_url}/league`, method: 'POST', body: {
        name,
        description: '',
        team_size
      }});
      return response.result[0];
    });
}

async function getOrCreatePlayers(player_count) {
  let users = [];
  for (let i = 0; i < player_count; i+=1) {
    const gen_player_username = i => config.env + '-doublelim' + i;
    const player_username = gen_player_username(i);
    const player_password = PLAYER_PASSWORD;
    let user;
    try {
      user = await UserController.loginWithPassword(player_username, player_password, 'bebo');
    } catch(err) {
      if (err.code && err.code === '404') {
        user = await UserController.signupWithPassword(player_username, player_password, null, null, CAMPAIGN_NAME, 'bebo');
      } else {
        throw err;
      }
    }
    if (user) {
      users.push(user);
    }
  }
  return users;
}

async function addAllUsersToLeague(users, league) {
  const { league_id } = league;
  for (const user of users) {
    const { user_id } = user;
    const user_team = await UserTeamModel.findOne({ user_id, league_id }, false, true);
    const is_creator = users.indexOf(user) < (users.length / team_size);
    logger.debug({ user_team, username: user.username, is_creator });
    if (user_team) {
      logger.debug(user.username, 'has user_team, skippin');
      continue;
    }
    if (team_size === 1) {
      await addUserToLeague(user, league);
      continue;
    }

    if (is_creator) {
      logger.debug(user.username, 'creating team');
      await createTeamInLeague(user, league);
    } else {
      const invite_code = TEAM_INVITE_CODES.shift();
      logger.debug(user.username, 'joining team', invite_code);
      await addUserToTeam(user, invite_code);
    }
  }
}

async function addUserToLeague(user, league) {
  const { league_id } = league;
  const invite = await getOrCreateLeagueInvite(league_id);
  const { code } = invite;
  logger.info('Invite code:', code);
  await InviteController.acceptInviteByCode(code, user.user_id);
}

async function getOrCreateLeagueInvite(league_id) {
  const invite = await InviteModel.findAll2({ where: { league_id } });
  if (invite && invite.length) {
    return invite[0];
  }
  const body = { league_id, campaign_name: CAMPAIGN_NAME };
  const response = await doAdminRequest({ url: `${config.aws_api_url}/invite`, method: 'POST', body });
  return response.result[0];
}

async function createTeamInLeague(user, league) {
  const { user_id } = user;
  const { league_id } = league;
  const response = await doAdminRequest({
    url: `${config.aws_api_url}/team`,
    method: "POST",
    body: {
      league_id
    },
    headers: {
      'X-User-Id': user_id
    }
  });
  const split = response.result[0].invite_url.split("/");
  const code = split[split.length - 1];
  TEAM_INVITE_CODES.push(code);
  logger.debug({ TEAM_INVITE_CODES, code });
}

async function addUserToTeam(user, invite_code) {
  await InviteController.acceptInviteByCode(invite_code, user.user_id);
}

(async () => {
  // const league_name = 'Tourney League #' + Date.now();
  // const league_name = 'Tourney League #1549579141933';
  await config.init({name: "pharah"});
  await logger.init(config);
  await Broadcast.init(config);
  await Dispatcher.init(config);
  await Analytics.init();
  bracketBot = require('../bots/bracket_bot');
  stormBot = require('../bots/storm_bot');

  PLAYER_PASSWORD = await getPass("Player Password");

  let i = 1;
  logger.info(++i);

  const TarlyController = require('../classes/tarly_controller');
  await TarlyController.setupModels();
  logger.info(++i);

  const User = require('../api/user');
  User.init(config);
  LeagueModel = TarlyController.getModel('league');
  logger.info("league");
  InviteModel = TarlyController.getModel('invite');
  UserTeamModel = TarlyController.getModel('user_team');
  logger.info("user_team");
  UserController = require('../controllers/user');
  logger.info("user");
  InviteController = require('../controllers/invite');
  logger.info("invite");
  logger.info(++i);

  if (isNaN(player_count)) {
    logger.error('bruh das not a ######');
    process.exit(1);
  }

  if (!(player_count % team_size === 0)) {
    logger.error(`${player_count} cannot evenly fit into teams of ${team_size}`);
    process.exit(1);
  }
  logger.info(++i);

  LEAGUE_NAME = LEAGUE_NAME + ' - ' + player_count;
  logger.info(++i);

  const tournament_options = {
    delay,
    tournament_type
  };
  logger.info(++i);

  const users = await getOrCreatePlayers(player_count);
  logger.info(++i);
  const league = await getOrCreateLeague(LEAGUE_NAME);
  logger.info(++i);
  logger.info('League:', { league_name: league.name, league_id: league.league_id });
  await addAllUsersToLeague(users, league);
  let runBots = stormBot;
  if (tournament_type  === "double_elim" || tournament_type === "single_elim") {
    runBots = bracketBot;
  }
  await runBots(league.league_id, player_count, users.map(user => ({ username: user.username, password: PLAYER_PASSWORD }) ), tournament_options);
})();
