const request = require('request-promise');
const Promise = require('bluebird');

const config = require('../config');
const logger = require('../api/logger');
const readline = require('readline').createInterface({
  input: process.stdin,
  output: process.stdout
});

const BaseBot = require('./base_bot');

const ADMIN_HEADERS = {
  'X-Api-Key': config.SERVICE_API_KEY
};


const LOBBY_DURATION = 20; 
const END_BUFFER = 5;
// const ROUND_DURATION = 20 * 60;

const ROUND_DURATION = 120;
const MAX_KILLS = 50;
let max_kills = 50;
const MAX_KILLS_SEC = MAX_KILLS / (30 * 60) ;
max_kills = Math.min(max_kills, MAX_KILLS_SEC * ROUND_DURATION);


const TYPE = "storm";
const TOURNAMENT_TEMPLATE = {
  name: `Rivals`,
  type: TYPE,
  config: {
    rounds : [
      // { survivors: 500, seconds: 120},
      // { survivors: 250, seconds: 120},
      // { survivors: 100, seconds: 120},
      // { survivors: 50, seconds: ROUND_DURATION },
      // { survivors: 25, seconds: ROUND_DURATION },
      { survivors: 12, seconds: ROUND_DURATION },
      { survivors: 1, seconds: ROUND_DURATION }
    ]
  },
  points_needed: 0
};

async function doAdminRequest(passedOptions) {
  const headers = Object.assign({}, passedOptions.headers || {}, ADMIN_HEADERS);
  const options = Object.assign({}, passedOptions, { headers, json: true });
  logger.info(`${options.method} ${options.url}`, options.body || {});
  return request(options);
}

function getline(str) {
  return new Promise(resolve => {
    readline.question(str + ' ', answer => {
      resolve(answer);
    });
  });
}

async function startNewTourney(league_id, options) {

  let {tournament_type, delay}  = options;

  const response = await doAdminRequest({
    url: `${config.aws_api_url}/tournament?league_id=${league_id}&state=started&count=100`,
    headers: ADMIN_HEADERS,
    method: 'GET'
  });

  response.result.map(async tournament => {
    await doAdminRequest({ method: 'PUT', body: { tournament_id: tournament.tournament_id, state: 'ended', end_dttm: "$NOW" }, url: `${config.aws_api_url}/tournament` });
  });

  let tournament = Object.assign({}, TOURNAMENT_TEMPLATE, {
    league_id,
    start_dttm: Math.floor((Date.now() + delay) / 60000) * 60000,
  });

  await doAdminRequest({ url: `${config.aws_api_url}/league`, method: 'PUT', body: { league_id, active_tournament_id: null } });
  let tournamentResult = await doAdminRequest({ url: `${config.aws_api_url}/tournament`, method: 'POST', body: tournament});
  logger.info("created tournament", tournamentResult.result[0].tournament_id);

}

const ROUTES = [
  '/user',
  '/subscribe/user',
  '/subscribe/match',
  '/match',
  '/tournament',
  '/subscribe/tournament',
  '/league',
  '/subscribe/league',
];

class StormBot extends BaseBot {

  constructor(i, bot_count) {
    super();
    this.kill_cnt = i;
    this.victory_cnt = false;
    // if (i > bot_count/2) {
    //   this.victory_cnt = true;
    // }
    // this.kill_cnt = Math.random() * bot_count;
    if (bot_count > max_kills) {
      this.kill_cnt  = this.kill_cnt * max_kills / bot_count ;
    }
    this.game_cnt = 1;
    this.playing = false;
    this.victory_cnt = false;
    this.routes = new Set(ROUTES);
  }

  async login(user) {
    await super.login(user);
  }

  async playRound() {
    if (this.playing) {
      return;
    }
    this.playing = true;
    this.doPlayRound();
  }

  nextKillDelay(cut_off, kill_cnt) {
    let kill_delay = cut_off - new Date();
    kill_delay -= END_BUFFER * 1000;
    if (kill_cnt < 1) {
      kill_cnt = 1;
    }
    return kill_delay / kill_cnt;
  }

  async doPlayRound() {

    let match_id = this.match.match_id;
    let tournament_id = this.match.tournament_id;
    let tournament = this.tournaments[tournament_id];
    let cut_off = new Date(this.set.started_dttm) + (ROUND_DURATION * 1000 / this.game_cnt);
    if (tournament != null && tournament.next_round_dttm != null) {
      cut_off = new Date(tournament.next_round_dttm);
    }

    try {
      await this.startGame();
      if (this.kill_cnt) {
        for (let i = 0; i < this.kill_cnt; i += 1) {
          let kill_delay = this.nextKillDelay(cut_off, this.kill_cnt-i);
          logger.debug(this.user.username, "waiting", kill_delay, cut_off, this.kill_cnt-i);
          await Promise.delay(kill_delay);
          if (match_id != this.match.match_id) {
            logger.debug(this.user.username, "exit game loop, match changed");
            return;
          }
          await this.kill();
        }
      }
      if (match_id != this.match.match_id) {
        logger.debug(this.user.username, "exit game loop, match changed");
        return;
      }
      if (this.victory_cnt)  {
        await this.victory();
      }
      await Promise.delay(Math.random() * END_BUFFER * 1000);
      await this.finishGame();

    } catch (err) {
      logger.error(err);
    }
  }

  async doFSM(entity) {
    logger.debug(this.user.username, JSON.stringify(entity, null, 2));
    let oldState = this.state;
    let old_match_id = this.match && this.match.match_id;
    if (entity.url === '/match') {
      this.match = entity;
      this.set = this.getMySet(entity);
    }
    logger.info(this.user.username, 'doFSM', this.state, entity.url, this.set && this.set.state);
    let match_id = this.match.match_id;

    if (entity.url === '/league') {
      this.leagues[entity.league_id] = entity;
    } else if (entity.url === '/tournament') {
      this.tournaments[entity.tournament_id] = entity;
    } else if (entity.url === '/match' && entity.deleted_dttm) {
      await this.goIdle();
    } else if (entity.url === '/user') {
      if (entity.active_match_id) {
        match_id = entity.active_match_id;
        if (old_match_id !== match_id) {
          this.state = 'idle';
          this.playing = false;
          logger.info(this.user.username, 'Enter new match', match_id);
        }
        await this.subscribeMatch(entity.active_match_id);
      } 
      for (const league of entity.leagues) {
        await this.subscribeLeague(league.league_id);
        await this.checkin(league);
      }
    } else if (this.state === 'idle') {
      if (entity.url === '/match') {
        if (this.set.state === 'waiting') {
          this.state = 'waiting';
          logger.info(this.user.username, 'Getting ready for match', this.match.match_id);
          await this.simulateLobbyReady();
        } else if (this.set.state === 'started') {
          this.state = 'game';
          this.round = this.set.round;
          if (this.round < this.game_cnt) {
            logger.info(this.user.username, 'Playing round', this.round);
            await this.playRound(entity);
          }
        }
      }
    } else if (this.state === 'waiting') {
      if (entity.url === '/match') {
        if (this.set.state === 'started') {
          this.state = 'game';
          this.round = this.set.round;
          if (this.round < this.game_cnt) {
            logger.info(this.user.username, 'Playing round', this.round);
            await this.playRound(entity);
          }
        }
        if (this.set.state === 'waiting') {
          logger.info(this.user.username, 'Getting ready for match (again)', this.match.match_id);
          await this.simulateLobbyReady();
        }
      }
    } else if (this.state === 'game') {
      if (this.set.state === 'started') {
        if (this.round < this.set.round && this.set.round < this.game_cnt) {
          this.round = this.set.round;
          logger.info(this.user.username, 'Playing round', this.round);
          await this.playRound(entity);
        }
      } else if (this.set.state === 'ended') {
        logger.info(this.user.username, entity.match_id, this.set.set_id, 'set ended');
        this.goIdle();
      }
    } else {
      logger.info(this.user.username, 'unknown handled state messate', entity);
    }

    if (this.state !== oldState) {
      if (this.match.match_id) {
        match_id = this.match.match_id;
      }
      logger.info(this.user.username, match_id, 'state transition', oldState, '->', this.state);
    }
  }
}

async function run(league_id, bot_count, logins, options) {
  await startNewTourney(league_id, options);
  await getline('start bots ? (enter)');
  for (let i = 0; i < bot_count; i++) {
    try {
      const bot = new StormBot(i, bot_count);
      logger.info('created bot', i);
      await bot.login(logins[i]);
      await bot.start();
    } catch(err) {
      logger.error(`Bot #${i} error`, err);
      logger.error('Unable to start bot', i);
    }
  }
}

module.exports = run;
