const BeboNodeWS = require('bebo-node-ws');
const logger = require('../api/logger');
const config = require('../config');
const uuidv4 = require('uuid/v4');

class WebSocketClient {
  constructor(user, connectCallback) {
    this.reqIdMap = {};
    this.messageListener = () => logger.error('override messageListener');
    this.user = user;
    this.name = user.username;
    this.connectCallback = connectCallback;
    this.connect();
  }

  connect() {
    logger.info('connecting to:', config.RAVEN_URL);
    this.client = new BeboNodeWS(config.RAVEN_URL);
    this.client.addEventListener('open', this.onOpen.bind(this));
    this.client.addEventListener('message', this.onMessage.bind(this));
    this.client.addEventListener('error', this.onError.bind(this));
    this.client.addEventListener('close', this.onError.bind(this));
  }

  onOpen() {
    this.subscribe();
  }

  subscribe(retry = 0) {
    if (retry === 3) {
      logger.error('HELP FIX RAVEN PLZZZ');
      setTimeout(() => this.subscribe(), 60 * 1000);
      return;
    }

    this.wsSend(
      {
        url: '/subscribe/user',
        method: 'POST',
        access_token: this.user.access_token,
        req_id: uuidv4()
      },
      message => {
        const { code } = message;
        if (code === 200) {
          this.connectCallback(message);
        } else {
          logger.error('Tinder subscribe error', retry);
          setTimeout(() => this.subscribe(retry + 1), 3000);
        }
      }
    );
  }

  onMessage(message) {
    message = message.data;
    // logger.debug(this.name, 'WS onMessage', message, typeof message);
    if (message === 'ping') {
      return;
    }

    try {
      const json = JSON.parse(message);
      const { url, ack_id } = json;

      if (!url) {
        logger.error('no url in websocket message', message);
        return;
      }

      if (ack_id && ack_id in this.reqIdMap) {
        // logger.info('WebSocketClient onMessage ack_id && in reqIdMap', json);
        const func = this.reqIdMap[ack_id];
        delete this.reqIdMap[ack_id];
        return func(json);
      }

      this.messageListener(json, this);
    } catch (err) {
      logger.warn('Failed to recieve websocket message', err && err.stack);
    }
  }

  setMessageListener(func) {
    this.messageListener = func;
  }

  onClose() {
    logger.warn('WS close');
  }

  onError(err) {
    logger.error(this.name, 'WS Error', err);
  }

  wsSend(payload, callback) {
    // logger.info('>>> wsSend', payload);
    let req_id = uuidv4();
    // logger.info('wsSend req_id', req_id);
    if (callback) {
      // logger.info('wsSend callack');
      payload.req_id = req_id;
      this.reqIdMap[req_id] = callback;
    }

    this.client.send(JSON.stringify(payload), err => {
      if (err) {
        logger.error(err);
        if (callback) {
          callback(err);
          delete this.reqIdMap[req_id];
        }
      }
    });
  }

  close() {
    if (this.client) {
      this.client.close();
    }
  }
}

module.exports = WebSocketClient;
