const _ = require('lodash');
const os = require('os');
const url = require('url');
const uuid = require('uuid');

const config = require('../config.js');
const logger = require('../api/logger');
let User = require('../api/user');

const Dispatcher = require('bebo-node-commons').Dispatcher;
const { makeESFriendly, makeUserESFriendly, getDefaultESEventFromRequest } = Dispatcher.Utils;

let clientEventQueue = null;
let apiAuditEventQueue = null;
let psqlAuditQueue = null;
let msgAuditQueue = null;

function init() {
  clientEventQueue = Dispatcher.getInstance({
    pickup_rate_ms: 30 * 1000,
    queueName: 'client-events_es'
  });

  apiAuditEventQueue = Dispatcher.getInstance({ 
    pickup_rate_ms: 30 * 1000,
    queueName: config.CODE_NAME + '-api_audit_es' 
  });

  psqlAuditQueue = Dispatcher.getInstance({
    queueName: config.CODE_NAME + '-psql_audit_es_',
    pickup_rate_ms: 30 * 1000
  });

  msgAuditQueue = Dispatcher.getInstance({
    queueName: 'msg-audit_es',
    pickup_rate_ms: 30 * 1000
  });
}

var hostname = os.hostname();

function mkClientEvent(category, action, label, value, options) {

  if (!options) {
    options = {};
  }

  if (label != null && typeof label !== 'string') {
    label = label.toString();
  }

  if (label == null) {
    label = undefined;
  }

  var now = new Date();

  var event = {
    _id: options._id || uuid.v4(),
    category_tx: category,
    action_tx: action,
    label_tx: label,
    event_at: now.getTime(),
    event_dttm: now.toISOString(),
    node_tx: hostname
  };

  if (label) {
    event.routing_key = category + '.' + action + '.' + label;
  } else {
    event.routing_key = category + '.' + action;
  }

  if (typeof value === 'number') {
    event.value_nr = value;
  }
  if (options) {
    _.assign(event, options);
    delete event.user;
    delete event.data;
    delete event.user_id;
  }
  if (options.user) {
    event.user = makeUserESFriendly(options.user);
    event.user_id = event.user.user_id;
  }

  if (options.data) {
    event.data = makeESFriendly(options.data, 1);
  }
  return event;
}
module.exports.mkClientEvent = mkClientEvent;

module.exports.track = async function track(category, action, label, value, options) {
  if (!category || !action) {
    return false;
  }
  if (options.user_id) {
    options.user = await User.get(options.user_id);
  }
  var event = mkClientEvent(category, action, label, value, options);

  // logger.debug("EVENT", JSON.stringify(event, null, 2));
  return clientEventQueue.writeEvent(event);
};


const WRITE_METHODS = new Set(['PUT', 'POST', 'DELETE']);

function mkApiAuditEventEs(req, res) {

  var ended = new Date();

  let path = req.originalPath;
  let u = url.parse(req.originalUrl);

  var r = '';
  if (path) {
    r = path
      .replace(/\//g, '.')
      .replace(/\.\.+/, '.');
  }
  r = _.trim(r, '.');
  var routing_key = req.method + '.' + r;

  var event = getDefaultESEventFromRequest(req);
  event.method_tx = req.method;
  event.routing_key = routing_key;
  event.url_tx = req.originalUrl;
  event.path_tx = u.pathname;
  event.domain_tx = req.hostname;
  event.query_tx = u.query;
  event.elapsed_ms = ended.getTime() - req._audit_event.event_at;
  event.event_at = req._audit_event.event_at;
  event.event_dttm = req._audit_event.event_dttm;
  event.ended_at = ended.getTime();
  event.ended_dttm = ended.toISOString();
  event.service_tx = config.CODE_NAME;

  if (req.body && 
    typeof(req.body) === 'object' && 
    Object.keys(req.body).length
  ) {
    let body_json = Object.assign({}, req.body);
    if (body_json.password) {
      body_json.password = "**hidden**";
    }
    event.body_tx = JSON.stringify(body_json, null, 2);
  }


  if (res) {

    event.status_code = res.statusCode;
    event.status_tx = res.statusMessage;

    if (res.result) {
      event.result_cnt = res.result.length;
      if (WRITE_METHODS.has(req.method) &&
        res.result.length === 1) {
        event.result = res.result[0];
      }
    }
    if (res.error_message) {
      event.message_tx = res.error_message;
    }
  }

  // if (res.statusCode >= 400) {
  event.headers = {};
  for (let [k,v] of Object.entries(req.headers)) {
    let newKey = k.replace(/-/g,"_").toLowerCase() + "_tx";

    if (newKey === "x_api_key_tx" || newKey === "x_access_token_tx") {
      if (v != null && v.length > 12) {
        v = v.replace(/.{12}$/, "************");
      }
    }
    event.headers[newKey] = v;
  }
  // }

  _.assign(event, req._audit_event);
  event = makeESFriendly(event, 1);
  event.user = makeUserESFriendly(req.acting_user);

  if (event.result_nest && event.result_nest.access_token_tx) {
    event.result_nest.access_token_tx = "**hidden**";
  }
  // logger.debug("ES EVENT", JSON.stringify(event, null, 2));

  return event;
}

module.exports.apiAudit = function apiAudit(req, res, callback) {
  var event = mkApiAuditEventEs(req, res);
  apiAuditEventQueue.writeEvent(event);
  callback();
};

module.exports.streamPointsAudit = () => {
  //CURRENTLY FULLY DISABLED
  // wtf removing breaks puts /match
  return Promise.resolve();
};

module.exports.psqlAudit = () => { return ;};

module.exports.msgAudit = message => {
  const event = makeESFriendly(Object.assign({}, message, { _id: message.sid }));
  msgAuditQueue.writeEvent(event);
};
module.exports.init = init;
