var _ = require('lodash');
var config = require('../config.js');

var HUMAN_ENDING = {
  plain: { 1000: 'k', 1000000: 'm' }
};

var Helper = {
  shortName: function shortName(user) {
    var first_name = user.first_name || '';
    var last_name = user.last_name || '';
    if (last_name === '' && first_name === '') {
      first_name = user.fullname;
    }

    if (first_name && first_name.length > 3 && last_name && last_name.length > 0) {
      return first_name + ' ' + last_name[0] + '.';
    } else {
      var name = first_name + ' ' + last_name;
      return _.trim(name);
    }
  },

  cookieName: function cookieName(name) {
    return name + '_' + config.cookie_env;
  },

  oldCookieName: function oldCookieName(name) {
    if (config.env === 'bebo-prod') {
      return name;
    }
    return name + '_' + config.cookie_env;
  },

  getMobileOperatingSystem: function getMobileOperatingSystem(req) {
    var userAgent = req.get('User-Agent');
    if (!userAgent) {
      return 'unknown';
    }

    if (
      userAgent.match(/iPad/i) ||
      userAgent.match(/iPhone/i) ||
      userAgent.match(/iPod/i) ||
      userAgent.match(/CFNetwork/i)
    ) {
      return 'ios';
    } else if (userAgent.match(/Windows Phone|Windows CE/i)) {
      // return 'windows_phone';
      return 'web';
    } else if (userAgent.match(/Android/i)) {
      return 'android';
    } else if (userAgent.match(/Windows NT|\(X11; Linux|\(X11; Ubuntu|\(Macintosh;/i)) {
      return 'web';
    } else {
      // return 'unknown';
      return 'web';
    }
  },

  collectProps: function(props, key) {
    var s = new Set();
    var lists = _.values(props);
    lists.forEach(function(l) {
      if (!l) {
        return;
      }
      l.forEach(function(obj) {
        s.add(obj[key]);
      });
    });
    return Array.from(s);
  },

  isHost: function isHost(stream, user_id) {
    if (stream.creator && stream.creator.user_id === user_id) {
      return true;
    }
    if (stream.cohosts && stream.cohosts.indexOf(user_id) !== -1) {
      return true;
    }
    return false;
  },

  humanize: function(number, type) {
    var end = '';
    var n = number.toString();
    var fraction = '';

    if (!type) {
      type = 'plain';
    }
    if (number > 1000000) {
      end = HUMAN_ENDING[type][1000000];
      n = Math.floor(number / 1000000).toString();
      var f = number % 1000000;
      if (f !== 0) {
        fraction = f.toString();
      }
    } else if (number > 1000) {
      end = HUMAN_ENDING[type][1000];
      n = Math.floor(number / 1000).toString();
      var f = number % 1000;
      if (f !== 0) {
        fraction = f.toString();
      }
    }
    if (n.length > 1 || fraction.length === 0) {
      fraction = '';
    } else {
      fraction = '.' + fraction[0];
    }
    return n + fraction + end;
  }
};


if (!config.isProd()) {
  // TESTS:
  var assert = require('assert');
  function mockReq(ua) {
    return {
      get: function() {
        return ua;
      }
    };
  }
  var testUa = [
    [
      'android',
      'Mozilla/5.0 (Linux; Android 4.4.3; KFTHWI Build/KTU84M) AppleWebKit/537.36 (KHTML, like Gecko) Silk/47.1.79 like Chrome/47.0.2526.80 Safari/537.36'
    ],
    [
      'android',
      'Mozilla/5.0 (Linux; Android 5.0.2; LG-V410/V41020c Build/LRX22G) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/34.0.1847.118 Safari/537.36'
    ],
    [
      'android',
      'Mozilla/5.0 (Linux; Android 5.0.2; SAMSUNG SM-T550 Build/LRX22G) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/3.3 Chrome/38.0.2125.102 Safari/537.36'
    ],
    [
      'android',
      'Mozilla/5.0 (Linux; Android 5.1.1; SM-G928X Build/LMY47X) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/47.0.2526.83 Mobile Safari/537.36'
    ],
    [
      'android',
      'Mozilla/5.0 (Linux; Android 6.0.1; Nexus 6P Build/MMB29P) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/47.0.2526.83 Mobile Safari/537.36'
    ],
    [
      'android',
      'Mozilla/5.0 (Linux; U; Android 4.0.3; ko-kr; LG-L160L Build/IML74K) AppleWebkit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30'
    ],
    [
      'android',
      'Mozilla/5.0 (Linux; U; Android 4.2.2; he-il; NEO-X5-116A Build/JDQ39) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Safari/534.30'
    ],
    [
      'web',
      'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/41.0.2227.1 Safari/537.36'
    ],
    [
      'web',
      'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_2) AppleWebKit/601.3.9 (KHTML, like Gecko) Version/9.0.2 Safari/601.3.9'
    ],
    [
      'web',
      'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_3) AppleWebKit/537.75.14 (KHTML, like Gecko) Version/7.0.3 Safari/7046A194A'
    ],
    [
      'web',
      'Mozilla/5.0 (Macintosh; U; PPC Mac OS X 10_5_8; ja-jp) AppleWebKit/533.20.25 (KHTML, like Gecko) Version/5.0.4 Safari/533.20.27'
    ],
    [
      'web',
      'Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/41.0.2228.0 Safari/537.36'
    ],
    [
      'web',
      'Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/47.0.2526.111 Safari/537.36'
    ],
    ['web', 'Mozilla/5.0 (Windows NT 6.1; WOW64; rv:40.0) Gecko/20100101 Firefox/40.1'],
    [
      'web',
      'Mozilla/5.0 (Windows NT 6.3; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/41.0.2225.0 Safari/537.36'
    ],
    ['web', 'Mozilla/5.0 (X11; Linux i586; rv:31.0) Gecko/20100101 Firefox/31.0'],
    [
      'web',
      'Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/51.0.2704.106 Safari/537.36'
    ],
    ['web', 'Mozilla/5.0 (X11; Ubuntu; Linux x86_64; rv:15.0) Gecko/20100101 Firefox/15.0.1'],
    [
      'ios',
      'Mozilla/5.0 (iPad; CPU OS 6_0 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Version/6.0 Mobile/10A5355d Safari/8536.25'
    ],
    [
      'ios',
      'Mozilla/5.0 (iPhone; CPU iPhone OS 8_0 like Mac OS X) AppleWebKit/600.1.4 (KHTML, like Gecko) Version/8.0 Mobile/12A366 Safari/600.1.4'
    ],
    [
      'ios',
      'Mozilla/5.0 (iPhone; CPU iPhone OS 8_0_2 like Mac OS X) AppleWebKit/600.1.4 (KHTML, like Gecko) Version/8.0 Mobile/12A366 Safari/600.1.4'
    ]
    // ["unknown", "Mozilla/5.0 (Nintendo WiiU) AppleWebKit/536.30 (KHTML, like Gecko) NX/3.0.4.2.12 NintendoBrowser/4.3.1.11264.US"],
    // ["unknown", "Mozilla/5.0 (PlayStation 4 3.11) AppleWebKit/537.73 (KHTML, like Gecko)"],
    // ["unknown", "Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)"],
    // ["unknown", "Mozilla/5.0 (compatible; Yahoo! Slurp; http://help.yahoo.com/help/us/ysearch/slurp)"],
    // ["unknown", "Mozilla/5.0 (compatible; bingbot/2.0; +http://www.bing.com/bingbot.htm)"],
    // ["windows_phone", "HTC_Touch_3G Mozilla/4.0 (compatible; MSIE 6.0; Windows CE; IEMobile 7.11)"],
    // ["windows_phone", "Mozilla/5.0 (Windows Phone 10.0; Android 4.2.1; Microsoft; Lumia 950) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/46.0.2486.0 Mobile Safari/537.36 Edge/13.10586"],
    // ["windows_phone", "Mozilla/5.0 (Windows Phone 10.0; Android 4.2.1; Xbox; Xbox One) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/46.0.2486.0 Mobile Safari/537.36 Edge/13.10586"],
    // ["windows_phone", "Mozilla/5.0 (compatible; MSIE 9.0; Windows Phone OS 7.5; Trident/5.0; IEMobile/9.0)"],
  ];
  testUa.forEach(function(testData) {
    var ua = testData[1];
    var platform = testData[0];
    assert.equal(
      Helper.getMobileOperatingSystem(mockReq(ua)),
      platform,
      'expected ' + platform + 'but got: ' + ua
    );
  });
}

module.exports = Helper;
