const Twilio = require('twilio');
const uuid = require('uuid/v4');
const googleLibPhoneNumber = require('google-libphonenumber');
const phoneUtil = googleLibPhoneNumber.PhoneNumberUtil.getInstance();
const PNF = googleLibPhoneNumber.PhoneNumberFormat;

const logger = require('../api/logger');
const config = require('../config');
const TarlyController = require('./tarly_controller');
const { msgAudit } = require('./analytics');

class SMS {
  constructor() {
    this.twilio_sdk = new Twilio(config.TWILIO_ACCOUNT_SID, config.TWILIO_AUTH_TOKEN);

    this.from = config.TWILIO_MSG_SERVICE_SID;

    this.upsertMessage = this.upsertMessage.bind(this);
    this.saveToKibana = this.saveToKibana.bind(this);
    this.sendMessage = this.sendMessage.bind(this);
    this.parsePhoneNumber = this.parsePhoneNumber.bind(this);
    this.msgAuditModel = TarlyController.getModel('msg_audit');
  }

  parsePhoneNumber(phone_number) {
    if (! phone_number) {
      return;
    }
    let phoneNumber = null;
    try {
      phoneNumber = phoneUtil.parseAndKeepRawInput(phone_number);
    } catch (err) {
      logger.debug(err);
    }

    if (!phoneNumber || !phoneUtil.isValidNumber(phoneNumber)) {
      try {
        phoneNumber = phoneUtil.parseAndKeepRawInput(phone_number, 'US');
      } catch (err) {
        logger.info("not a US phone number either", phone_number, err);
      }
    }

    if (phoneNumber) {
      if (phoneUtil.isValidNumber(phoneNumber)) {
        phoneNumber = phoneUtil.format(phoneNumber, PNF.E164);
      } else {
        phoneNumber = null;
      }
    }
    return phoneNumber;
  }

  saveToKibana(message) {
    msgAudit(message);
  }

  upsertMessage(message) {
    logger.debug('SMS: upsertMessage pre-transform', message);
    return Promise.resolve({
      msg_audit_id: message.msg_audit_id || uuid(),
      external_id: message.sid || message.SmsSid || message.MessageSid,
      type: 'twilio',
      body: message.body || message.Body,
      from: message.from || message.From,
      status: message.status || message.MessageStatus,
      to: message.to || message.To,
      account_sid: message.accountSid || message.AccountSid,
      api_version: message.apiVersion || message.ApiVersion,
      direction: message.direction,
      error_code: message.errorCode && parseInt(message.errorCode),
      error_message: message.errorMessage,
      messaging_service_sid: message.messagingServiceSid,
      num_media: message.numMedia && parseInt(message.numMedia),
      num_segments: message.numSegments && parseInt(message.numSegments),
      price: message.price && parseInt(message.price),
      price_unit: message.priceUnit,
      uri: message.uri,
      subresource_uris: message.subresourceUris,
      sent_dttm: message.sent_dttm || message.dateSent && new Date(message.dateSent),
      created_dttm: message.created_dttm || message.dateCreated && new Date(message.dateCreated),
      updated_dttm: message.updated_dttm || message.dateUpdated && new Date(message.dateUpdated)
    })
    .then(message => {
      logger.debug('SMS:upsertMessage post-transform', message);
      return message;
    })
    .then(message => Promise.all([
      this.msgAuditModel.model.upsert(message),
      this.saveToKibana(message)
    ]).then(() => message));
  }

  sendMessage(to, body, subject) {
    //TODO rate limit on the to field
    const options = {
      body,
      from: this.from,
      to,
      statusCallback: config.TWILIO_CALLBACK_URL
    };
    logger.debug('SMS:sendMessage options', options);
    return this.twilio_sdk.messages.create(options)
    .then(this.upsertMessage)
    .catch(err => {
      logger.error('Immediate Error sending Twilio message', err);
      //todo blacklist/handle diff err cases
      throw err;
    });
  }
}

module.exports = new SMS();
