const Sequelize = require('sequelize');

const config = require('../config.js');
const Logger = require('bebo-node-commons').Logger;
const log = require('../api/logger');

const { psqlAudit } = require('./analytics.js');


const psqlLogger = Logger.getLogger("pharah-sequelize");
psqlLogger.init(config);

const replication = { read: [], write: null };

replication.write = {
  host: config.DB_HOSTNAME,
  username: config.DB_USERNAME,
  password: config.DB_PASS,
  database: config.DB_NAME
};

config.DB_SLAVES.forEach(db_hostname => {
  const c = {
    host: db_hostname,
    username: config.DB_USERNAME,
    password: config.DB_PASS,
    database: config.DB_NAME
  };
  replication.read.push(c);
});

const EXCLUDED_TABLE_NAMES = ['challonge_match', 'challonge_participant', 'challonge_tournament'];
const Op = Sequelize.Op;
const operatorsAliases = {
  $eq: Op.eq,
  $ne: Op.ne,
  $gte: Op.gte,
  $gt: Op.gt,
  $lte: Op.lte,
  $lt: Op.lt,
  $not: Op.not,
  $in: Op.in,
  $notIn: Op.notIn,
  $is: Op.is,
  $like: Op.like,
  $notLike: Op.notLike,
  $iLike: Op.iLike,
  $notILike: Op.notILike,
  $regexp: Op.regexp,
  $notRegexp: Op.notRegexp,
  $iRegexp: Op.iRegexp,
  $notIRegexp: Op.notIRegexp,
  $between: Op.between,
  $notBetween: Op.notBetween,
  $overlap: Op.overlap,
  $contains: Op.contains,
  $contained: Op.contained,
  $adjacent: Op.adjacent,
  $strictLeft: Op.strictLeft,
  $strictRight: Op.strictRight,
  $noExtendRight: Op.noExtendRight,
  $noExtendLeft: Op.noExtendLeft,
  $and: Op.and,
  $or: Op.or,
  $any: Op.any,
  $all: Op.all,
  $values: Op.values,
  $col: Op.col
};

const options = {
  replication: replication,
  dialect: 'postgres',
  /* logging: psqlLogger.info, */
  logging: (query, bm, details) => {
    try {
      let tables = [];
      if (details && details.tableNames) {
        tables = details.tableNames;
      }
      const table = tables[0];
      if (table && EXCLUDED_TABLE_NAMES.indexOf(table) > -1) {
        return;
      }
      
      if (bm > 50) {
        psqlLogger.info(query, bm, "ms");
      }
      psqlAudit('pool', 'query', tables && tables.length === 1 ? tables[0] : '', bm, {
        query_tx: query,
        tables
      });
    } catch (err) {
      psqlLogger.error('Failed to log PSQL??', err);
    }
  },
  benchmark: true,
  pool: {
    max: 25,
    min: 5,
    idle: 20000,
    acquire: 8000
  },
  dialectOptions: {
    keepAlive: true,
    statement_timeout: 20000
  },
  operatorsAliases
};

const sql = new Sequelize(options);

sql.authenticate().then(errors => {
  if (errors) {
    log.error('failed to authenticated to pharah db', errors);
  } else {
    log.info('successfully authenticated to pharah db');
  }
});

const ssaclAttributeRoles = require('ssacl-attribute-roles');
ssaclAttributeRoles(sql);

module.exports = sql;
