const config = require('../config.js');
const logger = require('../api/logger');

const TarlyHandlerCreator = require('./tarly_handler');
const TarlyModelCreator = require('./tarly_model');
const PostgresQueryGenerator = require('sequelize/lib/dialects/postgres/query-generator');

class TarlyController {

  constructor() {
    //normalized path to the model config files
    const normalizedPath = require('path').join(process.cwd(), config.TARLY_MODEL_DIR);
    //array of config paths, model_names & routes, that we will loop over to set up the models
    //we filter out any filepath that does not end with .js

    this.paths = require('fs')
      .readdirSync(normalizedPath)
      .filter(path => path && path.endsWith('.js'))
      .map(path => {
        const configPath = require('path').join(process.cwd(), config.TARLY_MODEL_DIR, path);
        const modelName = path.replace('.js', '');
        return {
          configPath,
          modelName
        };
      });

    this.__models = {}; // sequelize models
    this._models = new Map(); // tarly models
    this._handlers = new Map(); // tarly handlers

    this.initialize = this.initialize.bind(this);
    this.setupModels = this.setupModels.bind(this);
    this.setupRelations = this.setupRelations.bind(this);
    this.setupHandlers = this.setupHandlers.bind(this);
  }

  initialize(app) {
    //app is the express app, we need this to link routes for generated handlers
    this.app = app;
    return this;
  }

  async setupRelations() {
    for (let [tableName, model] of this._models) {
      try {
        if (model.config.relation) {
          for (let otherName in model.config.relation) {
            let otherModel = this._models.get(otherName);
            if (!otherModel) {
              logger.error('Invalid relation', tableName, ' -> ', otherName);
              continue;
            }
            for (let r of model.config.relation[otherName]) {
              let relation = Object.assign({}, r);
              if (relation.through && relation.through.table_name) {
                // n-to-m relation
                let throughName = relation.through.table_name;
                let throughModel = this._models.get(throughName);
                if (!throughModel) {
                  logger.error(
                    'Invalid relation(missing through)',
                    tableName,
                    ' -> ',
                    otherName,
                    relation
                  );
                  continue;
                }
                relation.through.model = throughModel.model;
                delete relation.through.table_name;
                model.model.belongsToMany(otherModel.model, relation);
                logger.debug(
                  'created relation',
                  tableName,
                  ' -> ',
                  otherName,
                  'via',
                  throughName,
                  model.pk,
                  r
                );
              } else if (relation.has_many) {
                delete relation.has_many;
                model.model.hasMany(otherModel.model, relation);
              } else {
                model.model.belongsTo(otherModel.model, relation);
              }
            }
          }
        }
        if (model.config.associate) {
          model.config.associate(this.__models);
        }
        this.fillModels(model.config.include);
        this.fillModels(model.config.emit);
      } catch (err) {
        logger.error('Error Setting Up Model for', tableName);
        throw new Error(err);
      }
    }
  }

  fillModels(obj, maxDepth = 0) {
    if (maxDepth > 5) {
      logger.warn('maxDepth exceeded', maxDepth);
      return;
    }
    maxDepth++;
    if (!obj) {
      return;
    }
    if (Array.isArray(obj)) {
      for (let o of obj) {
        this.fillModels(o, maxDepth);
      }
      return;
    }
    if (typeof obj !== 'object') {
      return;
    }
    for (let key in obj) {
      if (key === 'model') {
        continue;
      }
      let value = obj[key];
      this.fillModels(value, maxDepth);
    }
    if (obj.tableName) {
      let m = this._models.get(obj.tableName);
      if (m) {
        obj.model = m.model;
        obj.tarlyModel = m;
      }
      delete obj.tableName;
    }
  }

  async setupModels() {
    // we need to register sub/object routes befor wildcard routes:
    this.paths.sort((a, b) => b.modelName.length - a.modelName.length);
    this.paths.forEach(({ configPath, modelName }) => {
      try {
        //try to require the config file
        const modelConfig = require(configPath);
        //create the actual sequelize model out of the config
        const model = new TarlyModelCreator(modelConfig);
        //validate the name is correct (enforce filename match)
        model._validateName(modelName);
        //add the model into our map of models
        this._models.set(modelName, model);
        this.__models[modelName] = model.model;

      } catch (err) {
        //if we fail to set up the model we will log the error but otherwise continue
        logger.error(`Failed to set up Tarly Model: ${modelName}`, err, err && err.stack);
      }
    });
    return this.setupRelations();
  }

  setupHandlers() {
    return new Promise(resolve => {
      //iterate over all models
      this._models.forEach((model, modelName) => {
        try {
          const handler = new TarlyHandlerCreator(modelName, model, this.app);
          this._handlers.set(modelName, handler);
          if (modelName.indexOf('_') > -1) {
            const newRouteHandler = new TarlyHandlerCreator(modelName, model, this.app, true);
            this._handlers.set(`new_${modelName}`, newRouteHandler);
          }
        } catch (err) {
          //if we fail we will log the error but otherwise continue
          logger.error(`Failed to create Tarly Handler: ${modelName}`, err, err & err.stack);
        }
      });
      resolve();
    });
  }

  getModel(modelName) {
    if (this._models.has(modelName)) {
      return this._models.get(modelName);
    }
    throw new Error(`model not found: ${modelName}`);
  }

  getHandler(modelName) {
    if (this._handlers.has(modelName)) {
      return this._handlers.get(modelName);
    }
    throw new Error(`handler not found: ${modelName}`);
  }
}

const singleton = new TarlyController();

module.exports = singleton;
