const crypto = require('crypto');
const error = require('./error.js');
const logger = require('../api/logger');

const config = require('../config');
const Cache = require('bebo-node-commons').RedisCache;

const sequelize = require('sequelize');
const _ = require('lodash');
const async = require('async');

const cacheWrite = Cache.getWriteClient(config.TARLY_CACHE);
const cacheRead = Cache.getReadClient(config.TARLY_CACHE);

class TarlyHandlerCreator {

  constructor(modelName, model, app, new_routes) {
    this.app = app;
    this.model = model;
    this.modelName = modelName;

    //the base route for the handler is always generated out of the model name
    this.baseRoute = new_routes ? `/${modelName}` : `/${modelName.replace(/_/g, '/')}`;
    this.createRoutes = this.createRoutes.bind(this);
    this.getByAuth = this.getByAuth.bind(this);
    this.getById = this.getById.bind(this);
    this.getCachedResult = this.getCachedResult.bind(this);
    this.getList = this.getList.bind(this);
    this.emitEvents = this.emitEvents.bind(this);
    this.postByOwner = this.postByOwner.bind(this);
    this.putById = this.putById.bind(this);
    this.deleteById = this.deleteById.bind(this);
    this._handleResult = this._handleResult.bind(this);

    this.createRoutes();
  }

  _handleResult(result, forceList = false, cache_key) {
    if (!forceList && this.model.config.useOwnerUserIdAsPrimaryKey && Array.isArray(result)) {
      result = result[0];
    }
    if (!result) {
      return Promise.reject(new error.NotFound(`${this.modelName}: not found`));
    }

    //if we are supposed to set a cache and have a cache key passed to us then we will set the cache
    if (this.model.config.cacheTime && cache_key) {
      return cacheWrite.setAsync(cache_key, JSON.stringify(result)).then(() => {
        cacheWrite.expireAsync(cache_key, this.model.config.cacheTime);
        return result;
      });
    }

    //otherwise we just continue with the result

    return Promise.resolve(result);
  }

  get(req, res, next) {
    let wares = [
      this.model._getGetAuth(),
      this.getCachedResult,
      this.model.config.useOwnerUserIdAsPrimaryKey ?  this.getByAuth: this.getList
    ];
    this.runMiddleWares(wares, req, res, next);
  }
  put(req, res, next) {
    let wares = [
      this.model._getPutAuth(),
      this.putById,
      this.emitEvents
    ];
    this.runMiddleWares(wares, req, res, next);
  }
  delete(req, res, next) {
    let wares = [
      this.model._getDeleteAuth(),
      this.deleteById,
      this.emitEvents,
    ];
    this.runMiddleWares(wares, req, res, next);
  }
  post(req, res, next) {
    let wares = [
      this.model._getPostAuth(),
      this.postByOwner,
      this.emitEvents
    ];
    this.runMiddleWares(wares, req, res, next);
  }

  runMiddleWares(wares, req, res, next) {
    wares = _.flattenDeep(wares);
    async.each(wares, (fn, callback) => {
      fn(req, res, callback);
    }, (err) => {
      next(err);
    });
  }



  createRoutes() {
    //GETs
    // logger.info(`registering tarly GET: ${this.baseRoute}`);
    //
    if (!this.app.isReservedRoute(this.baseRoute)) {
      // logger.info(`routing_key GET${this.baseRoute.replace(/\//g, '.')}`);
      this.app.get(
        this.baseRoute,
        this.model._getGetAuth(),
        this.getCachedResult,
        this.model.config.useOwnerUserIdAsPrimaryKey ? this.getByAuth : this.getList,
      );
    }

    if (this.baseRoute.split('/').length === 2) {
      // top level
      let entityName = this.baseRoute.split('/')[1];
      let route = `/subscribe/${entityName}`;
      if (!this.app.isReservedRoute(route)) {
        this.app.get(
          route,
          this.model._getGetAuth(),
          this.getCachedResult,
          this.getById,
        );
      }
    }

    // logger.info(`registering tarly GET: ${this.baseRoute + '/:id'}`);
    if (!this.app.isReservedRoute(this.baseRoute + '/:id')) {
      this.app.get(
        this.baseRoute + '/:id',
        this.model._getGetAuth(),
        this.getCachedResult,
        this.getById,
      );
    }

    //POST
    // logger.info(`registering tarly POST: ${this.baseRoute}`);
    if (!this.app.isReservedRoute(this.baseRoute)) {
      // logger.info(`routing_key POST${this.baseRoute.replace(/\//g, '.')}`);
      this.app.post(
        this.baseRoute,
        this.model._getPostAuth(),
        this.postByOwner,
        this.emitEvents
      );

      // logger.info(`routing_key PUT${this.baseRoute.replace(/\//g, '.')}`);
      this.app.put(
        this.baseRoute,
        this.model._getPutAuth(),
        this.putById,
        this.emitEvents
      );
    }

    if (!this.app.isReservedRoute(this.baseRoute + '/:id')) {
      //PUT
      // logger.info(`registering tarly PUT: ${this.baseRoute + '/:id'}`);
      // logger.info(`routing_key PUT${this.baseRoute.replace(/\//g, '.')}`);
      this.app.put(
        this.baseRoute + '/:id',
        this.model._getPutAuth(),
        this.putById,
        this.emitEvents
      );
    }


    if (!this.app.isReservedRoute(this.baseRoute + '/:id')) {
      //DELETE
      // logger.info(`routing_key DELETE${this.baseRoute.replace(/\//g, '.')}`);
      // logger.info(`registering tarly DELETE: ${this.baseRoute + '/:id'}`);
      this.app.delete(
        this.baseRoute + '/:id',
        this.model._getDeleteAuth(),
        this.deleteById,
        this.emitEvents
      );
    }
    this.app.reserveRoute(this.baseRoute);
  }

  getCachedResult(req, res, next) {
    //if there is no cache time set, then just continue
    if (!this.model.config.cacheTime || (req.query && req.query.no_cache)) {
      return next();
    }
    if (!req.params) {
      req.params = {};
    }
    if (!req.query) {
      req.query = {};
    }
    //otherwise we're going to check the tarly cache
    const cache_key = crypto
      .createHash('md5')
      .update(`${this.modelName}_${JSON.stringify(req.query)}_${JSON.stringify(req.params)}`)
      .digest('hex');
    res.cache_key = cache_key;
    res.merged = {
      cache_key
    };

    return cacheRead
      .getAsync(res.cache_key)
      .then(result => {
        if (result) {
          res.result = JSON.parse(result);
        }
        next();
      })
      .catch(err => {
        logger.error('Tarly failed to get data from cache, continuing on to fetch data from DB', err);
        next();
      });
  }


  async emitEvents(req, res, next) {
    let result = res.result;
    if (! Array.isArray(result)) {
      result = [ result ];
    }

    if (this.model.config.emit) {
      for (let emit of this.model.config.emit) {
        for (let entity of result) {
          let id = entity[emit.foreignKey];
          if (!id) {
            logger.error("cant determine foreignKey", emit, "for",  entity);
            continue;
          }
          if (emit.through) {
            if (! emit.through.model ||
                ! emit.otherKey || 
                ! emit.foreignKey) {
              logger.error("emit.through needs model, otherKey and foreignKey", emit);
              continue;
            }

            let where = {};
            where[emit.foreignKey] = id;
            
            let relations = await emit.through.model.findAll({where});
            for (let r of relations) {
              let rId = r[emit.otherKey];
              await emit.tarlyModel.emit(rId);
            }
          } else {
            await emit.tarlyModel.emit(id);
          }
          // logger.error(emit);
        }
      }
    }
    next();
  }

  getByAuth(req, res, next) {
    if (res.result) {
      return next();
    }
    const showDeleted = req.query.show_deleted || false;
    return this.model
      .findOneByAuth(req, showDeleted)
      .then(result => this._handleResult(result, false, res.cache_key))
      .then(result => {
        res.result = result;
        next();
      })
      .catch(err => {
        next(new error.DefaultError(err));
      });
  }

  getById(req, res, next) {
    if (res.result) {
      return next();
    }
    const showDeleted = req.query.show_deleted || false;

    let id = req.params.id;
    if (!id) {
      id = req.query[this.model.pk];
    }
    return this.doGetById(id, showDeleted, req, res, next);
  }

  doGetById(id, showDeleted, req, res, next) {
    return this.model
      .findById(id, showDeleted)
      .then(result => this._handleResult(result, false, res.cache_key))
      .then(result => {
        res.result = result;
        next();
      })
      .catch(err => {
        next(new error.DefaultError(err));
      });
  }

  getList(req, res, next) {

    // FIXME delete this
    if (res.result) {
      logger.error("res.result - this should not happen");
      return next();
    }

    if (req.query[this.model.pk]) {
      const showDeleted = req.query.show_deleted || false;
      return this.doGetById(req.query[this.model.pk], showDeleted, req, res, next);
    }

    return this.model
      .findAll2(req.sqlOptions)
      // TODO look at inconsistent use of handleResult
      .then(result => this._handleResult(result, false, res.cache_key))
      .then(result => {
        res.result = result;
        next();
      })
      .catch(err => {
        next(new error.DefaultError(err));
      });
  }

  postByOwner(req, res, next) {
    return this.model
      .createByAuth(req)
      .then(this._handleResult)
      .then(result => {
        res.result = result;
        next();
      })
      .catch(sequelize.UniqueConstraintError, err => {
        logger.info('unique conflict', err);
        return next(new error.Conflict('DB Conflict'));
      })
      .catch(err => {
        next(new error.DefaultError(err));
      });
  }

  putById(req, res, next) {
    return this.model
      .updateById(req.params.id, req.body)
      .then(this._handleResult)
      .then(result => {
        res.result = result;
        next();
      })
      .catch(err => {
        next(new error.DefaultError(err));
      });
  }

  deleteById(req, res, next) {
    // FIXME - we want to return the soft deleted model ...
    return this.model
      .deleteByIdAndReturn(req.params.id)
      .then(this._handleResult)
      .then(() => {
        res.result = [];
        next();
      })
      .catch(err => {
        next(new error.DefaultError(err));
      });
  }
}

module.exports = TarlyHandlerCreator;
