const uuid = require('uuid');
const uuidv4 = require('uuid/v4');
const Broadcast = require('../api/broadcast');
const logger = require('../api/logger');
const Sequelize = require('sequelize');
const ssaclAttributeRoles = require('ssacl-attribute-roles');

const OwnerAuth = require('./owner_auth');
const TarlySQLPool = require('./sql_pool');

const BASE_MODEL_CONFIG = {
  paranoid: true,
  timestamps: true,
  createdAt: 'created_dttm',
  updatedAt: 'updated_dttm',
  deletedAt: 'deleted_dttm',
  freezeTableName: true
};

const BASE_AUTH = OwnerAuth;

class TarlyModelCreator {
  constructor(config) {
    this.config = config;
    if (!this.config) {
      throw new Error(`TarlyModelCreator::missing config object: ${this.config}`);
    }
    if (!this.config.table_name) {
      throw new Error(`TarlyModelCreator::missing model name: ${this.config.table_name}`);
    }

    this.pk = `${this.config.table_name}_id`;
    this.model = this._createSequelizeModelFromConfig(config);
  }

  findDeletedById(id) {
    return this.model.findByPk(id, { paranoid: false });
  }

  findById(id, showDeleted, raw = false) {
    if (this.config.include && this.config.include.length > 0) {
      return this.model.findByPk(id, {
        include: this.config.include,
        paranoid: !showDeleted,
        raw: false });
    }
    return this.model.findByPk(id, { paranoid: !showDeleted, raw });
  }

  findByPk(id, showDeleted, raw = false) {
    return this.model.findByPk(id, { paranoid: !showDeleted, raw });
  }

  entityToPayload(entity) {
    let url = "/" +  this.config.table_name.replace(/_/g, '/');
    let msg = {
      url,
    };

    let id = entity[this.pk];
    if (! id) {
      logger.error("entity", url, "is missing primary key", this.pk, "element", entity);
    }
    msg[this.pk] = id;

    let top_id_key = this.getTopLevelIdKey();
    let top_id = entity[top_id_key];
    if (! top_id) {
      logger.error("entity", url, "is missing top level key", top_id_key, "element", entity);
    }
    msg[top_id_key] = top_id;

    msg.event_id = uuidv4();
    msg.result = [entity] ;
    msg.current_at = Date.now();
    return msg;
  }

  getTopLevelIdKey() {
    return this.config.table_name.split('_')[0] + "_id";
  }

  async emit(id) {
    let opts = {paranoid: true, plain: true, useMaster: true};
    if (this.config.include) {
      opts.include = this.config.include;
    }
    let entity = await this.model.findByPk(id, opts);
    if (! entity) {
      return;
    }
    let payload = this.entityToPayload(entity);
    let url = payload.url;
    let top_id_key = this.getTopLevelIdKey();
    let to_id = payload[top_id_key];

    // the id must be the top level id - not the id of the element
    return Broadcast.emit(url, to_id, payload)
      .then(data => {
        logger.debug('Broadcast.emit success', payload, data, url, to_id);
      })
      .catch(err => {
        logger.error('Broadcast.emit', err, url, to_id);
      });
  }

  findManyByIds(ids, showDeleted, raw = false) {
    let where = {};
    where[this.pk] = { [Sequelize.Op.in]: ids };

    return this.model.findAll({
      where,
      paranoid: !showDeleted,
      raw
    });
  }

  findOne(where, showDeleted, raw = false) {
    return this.model.findOne({ where, paranoid: !showDeleted, raw });
  }

  findOneByAuth(req, showDeleted) {
    // FIXME
    logger.warning("findDO WE USE THIS???");
    const owner_id = req.acting_user.user_id;
    return this.findOne({ owner_id }, showDeleted);
  }

  async findAll2(opts) {
    let a = await this.model.findAll(opts);
    return a;
  }

  findAll(offset = 0, limit = 20, orderBy = null, showDeleted = false, raw = false) {
    return this.findAllWhere(null, offset, limit, orderBy, showDeleted, raw);
  }

  findAllWhere(where, offset = 0, limit = 20, orderBy, showDeleted = false, raw = false) {
    let opts = { offset, paranoid: !showDeleted, limit, raw };
    if (where) {
      opts.where = where;
    }
    if (orderBy) {
      opts.order = [orderBy];
    }
    return this.model.findAll(opts);
  }

  findAllByAuth(req, orderBy, showDeleted) {
    const owner_id = req.acting_user.user_id;
    const offset = req.query.offset || 0;
    const limit = req.query.count || 20;
    return this.findAllWhere({ owner_id }, offset, limit, orderBy, showDeleted);
  }

  create(createObj) {
    if (this.config.useOwnerUserIdAsPrimaryKey) {
      createObj[`${this.config.table_name}_id`] = createObj.user_id;
    } else if (
      !createObj[`${this.config.table_name}_id`] ||
      typeof createObj[`${this.config.table_name}_id`] !== 'string'
    ) {
      createObj[`${this.config.table_name}_id`] = uuid.v4();
    }

    if (!createObj.owner_id) createObj.owner_id = createObj[`${this.config.table_name}_id`];

    this.nowIsServerDate(createObj);
    return this.model.create(createObj, { raw: true });
  }

  bulkCreate(objArr) {
    return this.model.bulkCreate(objArr);
  }

  createByAuth(req) {
    const user_id = req.acting_user.user_id;
    const createObj = req.body;
    // FIXME: user_id here is really odd
    return this.create(Object.assign({}, createObj, { user_id, owner_id: user_id }));
  }

  nowIsServerDate(obj) {
    for(let [key, value] of Object.entries(obj)) {
      if (value === "$NOW" && key.endsWith("_dttm")) {
        obj[key] = Date.now();
      }
    }
  }

  updateById(id, updateObj, raw = true) {
    if (!id && updateObj[this.pk]) {
      id = updateObj[this.pk];
    }
    let where = {};
    where[this.pk] = id;
    this.nowIsServerDate(updateObj);
    return this.updateWhere(updateObj, where, raw, true);
  }

  updateWhere(updateObj, where, raw = true, single = false) {
    if (updateObj[this.pk]) {
      delete updateObj[this.pk];
    }
    this.nowIsServerDate(updateObj);
    return this.model
      .update(updateObj, { where, returning: true, raw })
      .then(([noOfRows, rows]) => {
        if (!noOfRows) {
          return null;
        }
        if (single) {
          return rows[0];
        }
        return rows;
      });
  }

  deleteById(id) {
    const where = {};
    where[this.pk] = id;
    return this.deleteWhere(where);
  }

  deleteByIdAndReturn(id) {
    const where = {};
    where[this.pk] = id;
    return TarlySQLPool.transaction(transaction => {
      return this.model.destroy({ where, transaction }).then(() => {
        return this.model.findByPk(id, { transaction, paranoid: false });
      });
    });
  }

  deleteWhere(where) {
    return this.model.destroy({ where });
  }

  _createSequelizeModelFromConfig(config) {
    const OWNER_ID_INDEX_NAME = `${config.table_name}_owner_id`;
    const MERGED_INDEXES = (config.table_indexes || []).filter(
      index => index.name !== OWNER_ID_INDEX_NAME
    );

    let BASE_MODEL = {};
    MERGED_INDEXES.concat({ name: OWNER_ID_INDEX_NAME, fields: ['owner_id'] });
    BASE_MODEL.owner_id = {
      type: Sequelize.STRING,
      allowNull: true
    };
    BASE_MODEL.url = {
      type: Sequelize.VIRTUAL
    };

    BASE_MODEL[`${config.table_name}_id`] = {
      type: Sequelize.STRING,
      allowNull: false,
      primaryKey: true
    };

    const MERGED_CONFIG = Object.assign(
      {
        getterMethods: {},
        setterMethods: {},
      },
      BASE_MODEL_CONFIG,
      config || {},
      { indexes: MERGED_INDEXES },
      config._config_overrides || {}
    );

    const urlValue = `/${config.table_name.replace(/_/g, '/')}`;
    MERGED_CONFIG.getterMethods.url = () => {
      return urlValue;
    };


    const sequelizeModel = TarlySQLPool.define(
      config.table_name,
      Object.assign({}, BASE_MODEL, config.table_schema),
      MERGED_CONFIG
    );

    ssaclAttributeRoles(sequelizeModel);
    return sequelizeModel;
  }

  _getGetAuth() {
    if (this.config.getAuth) {
      return [this.config.getAuth].concat(BASE_AUTH);
    }
    return BASE_AUTH;
  }

  _getPostAuth() {
    if (this.config.postAuth) {
      return [this.config.postAuth].concat(BASE_AUTH);
    }
    return BASE_AUTH;
  }

  _getPutAuth() {
    if (this.config.putAuth) {
      return [this.config.putAuth].concat(BASE_AUTH);
    }
    return BASE_AUTH;
  }

  _getDeleteAuth() {
    if (this.config.deleteAuth) {
      return [this.config.deleteAuth].concat(BASE_AUTH);
    }
    return BASE_AUTH;
  }

  _validateName(modelName) {
    if (modelName !== this.config.table_name) {
      throw new Error(
        `TarlyModelCreator::_validateName::invalid model name: ${
          this.config.table_name
        }, expected: ${modelName}`
      );
    }
  }
}

module.exports = TarlyModelCreator;
