const querystring = require('querystring');
const request = require('request-promise');
const logger = require('../api/logger');
const error = require('./error.js');

var config = require('../config.js');

const _baseUrl = 'https://api.twitch.tv/helix';
const authUrl = 'https://id.twitch.tv';
const authorizePath = '/oauth2/authorize';
const accessTokenPath = '/oauth2/token';
const timeout = 30 * 1000; //30 second timeout

class Twitch {
  constructor({ clientId, clientSecret, redirectUri }) {
    this.clientId = clientId;
    this.clientSecret = clientSecret;
    this.redirectUri = redirectUri;
    this.too_many_requests = false;
  }

  getScopes(isStreamer) {
    return isStreamer ? config.TWITCH_STREAMER_SCOPES : config.TWITCH_VIEWER_SCOPES;
  }

  validateTwitchCode(code) {
    let qs = {
      client_id: config.TWITCH_CLIENT_ID,
      client_secret: config.TWITCH_SECRET,
      code,
      grant_type: 'authorization_code',
      redirect_uri: config.TWITCH_REDIRECT_URI
    };
    const data = {
      method: 'POST',
      url: 'https://id.twitch.tv/oauth2/token',
      timeout: 20000,
      qs: qs,
      json: true
    };
    return request(data).then(body => {
      return body;
    });
  }

  validateAccessToken(token, logdata) {
    return request({
      method: 'GET',
      url: `https://id.twitch.tv/oauth2/validate`,
      headers: {
        Authorization: `OAuth ${token}`
      },
      json: true
    }).catch(err => {
      logger.error('twitch validation failed', err, logdata);
      return Promise.reject(new error.Unauthorized('token_revoked'));
    });
  }

  revokeAccessToken(token, logdata) {
    return request({
      method: 'POST',
      url: 'https://id.twitch.tv/oauth2/revoke',
      qs: { client_id: this.clientId, token: token },
      json: true
    }).catch(err => {
      logger.error('twitch revoke failed', err, logdata);
      return Promise.reject(new error.Unauthorized('token_revoked'));
    });
  }

  refreshToken(refresh_token, logdata) {
    if (!refresh_token) {
      logger.error('user has no refresh_token', logdata);
      return Promise.reject(new error.Unauthorized('token_revoked'));
    }
    let params = {
      grant_type: 'refresh_token',
      refresh_token: refresh_token,
      client_id: this.clientId,
      client_secret: this.clientSecret
    };

    let qs = querystring.stringify(params);
    return request({
      method: 'POST',
      url: `https://id.twitch.tv/oauth2/token?${qs}`,
      json: true
    }).catch(err => {
      logger.error('twitch refresh_token failed', err, logdata);
      return Promise.reject(new error.Unauthorized('token_revoked'));
    });
  }

  getAuthorizationUrl({ scopes, state, force_verify }) {
    let qs = { state, force_verify };
    let manual_scopes = '';
    if (!scopes) {
      logger.error('scopes missing');
    } else {
      manual_scopes = '&scope=' + scopes.join('+'); // twitch space
    }
    qs.response_type = 'code';
    qs.redirect_uri = this.redirectUri;
    qs.client_id = this.clientId;
    qs = querystring.stringify(qs);
    return `${authUrl}${authorizePath}?${qs}${manual_scopes}`;
  }

  getStreamsFromLogins(user_names_array) {
    const manualQueryString = user_names_array
      .filter(username => username && typeof username === 'string')
      .map((user_name, i) => {
        let part = '';
        if (i === 0) {
          part += '?';
        } else {
          part += '&';
        }
        part += `user_login=${user_name}`;
        return part;
      })
      .join('');

    const client_id_round_robin = [
      config.TWITCH_CLIENT_ID,
      '***REMOVED***',
      '***REMOVED***',
      '***REMOVED***',
      '***REMOVED***'
    ];

    if (this.too_many_requests) {
      logger.warn('skipping hitting twitch, we hit 429 recently');
      return Promise.reject('too many requests');
    }

    return request({
      method: 'GET',
      url: `https://api.twitch.tv/helix/streams${manualQueryString}`,
      json: true,
      headers: {
        'client-id': client_id_round_robin[Math.floor(Math.random() * client_id_round_robin.length)]
      }
    }).catch(err => {
      if (err || err.status === 429) {
        this.too_many_requests = true;
        setTimeout(() => {
          this.too_many_requests = false;
        }, 2 * 60 * 1000);
      }
      logger.error('twitch streamsFromLogin failed', err, err & err.stack);
      return Promise.reject(err);
    });
  }

  getUserLogins(user_names_array) {
    const manualQueryString = user_names_array
      .map((user_name, i) => {
        let part = '';
        if (i === 0) {
          part += '?';
        } else {
          part += '&';
        }
        part += `login=${user_name}`;
        return part;
      })
      .join('');

    const client_id_round_robin = [
      config.TWITCH_CLIENT_ID,
      '***REMOVED***',
      '***REMOVED***',
      '***REMOVED***',
      '***REMOVED***'
    ];

    if (this.too_many_requests) {
      logger.warn('skipping hitting twitch, we hit 429 recently');
      return Promise.reject('too many requests');
    }

    return request({
      method: 'GET',
      url: `https://api.twitch.tv/helix/users${manualQueryString}`,
      json: true,
      headers: {
        'client-id': client_id_round_robin[Math.floor(Math.random() * client_id_round_robin.length)]
      }
    }).catch(err => {
      if (err || err.status === 429) {
        this.too_many_requests = true;
        setTimeout(() => {
          this.too_many_requests = false;
        }, 2 * 60 * 1000);
      }
      logger.error('twitch streamsFromLogin failed', err, err & err.stack);
      return Promise.reject(err);
    });
  }
}

module.exports = Twitch;
