const Promise = require('bluebird');
const moment = require('moment');
const Sequelize = require('sequelize');

const TarlyController = require('../classes/tarly_controller');
const error = require('../classes/error.js');
const logger = require('../api/logger');
const TarlySQLPool = require('../classes/sql_pool');

const LeagueModel = TarlyController.getModel('league');
const DivisionModel = TarlyController.getModel('division');
const TeamModel = TarlyController.getModel('team');
const UserModel = TarlyController.getModel('user');
const TournamentModel = TarlyController.getModel('tournament');

const GET_MATCHES = `
WITH completed_matches AS (
       SELECT match.match_id,
              set.outcome,
              set.scores,
              set.team_id
       FROM match
                   JOIN match_set ON (match_set.match_id = match.match_id)
                   JOIN set ON (set.set_id = match_set.set_id)
       where match.state = 'ended'
         AND match.tournament_id IS NULL
         AND match.created_dttm > :time_since
         AND match.deleted_dttm IS NULL
         AND set.ended_dttm IS NOT NULL
         AND set.deleted_dttm IS NULL
)
SELECT team.team_id,
       completed_matches.match_id,
       completed_matches.scores,
       completed_matches.outcome
FROM team
  LEFT JOIN completed_matches ON (completed_matches.team_id = team.team_id)
WHERE team.active_division_id = :division_id
  AND team.deleted_dttm IS NULL
  AND (team.last_match_dttm > (NOW() - interval '31 days')
  OR team.created_dttm > (NOW() - interval '30 days'))
;`;


const calcPoints = (scores) => {
  let win = scores.win || 0;
  let loss = scores.loss || 0;
  let draw = scores.draw || 0;
  const games = win + loss + draw || 1;

  let points = 3 * win + draw;
  let score = scores.score || 0;
  points += score / (games * 109);
  logger.debug(scores, points, win, loss, draw, score, games);
  return points;
};

class DivisionController {
  async getAvailableDivision(league_id) {
    const divisions = await DivisionModel.findAll2({
      where: { league_id },
      useMaster: true
    });

    if (divisions.length === 0) {
      const new_division = await this.createDivision(league_id);
      return new_division;
    }

    return divisions[0].division_id;
  }

  async partitionDivision(division_id) {
    // TODO: break division into half
    return [];
  }

  async createDivision(league_id) {
    const league = await LeagueModel.model.findByPk(league_id, { paranoid: true, useMaster: true});
    if (!league) {
      throw new error.NotFound('League does not exists');
    }

    const division_number = Math.floor(Math.random() * 1000); // FIXME
    const division = await DivisionModel.create({
      league_id,
      name: `${league.name} ${division_number}`,
      image_url: league.image_url
    });

    return division;
  }
  async getDivisionStanding(division_id) {
    return TarlySQLPool.query(GET_MATCHES, {
      type: Sequelize.QueryTypes.SELECT,
      replacements: {
        division_id,
        time_since: moment()
          .tz('America/Los_Angeles')
          .startOf('month')
          .toISOString()
      }
    }).then(async matches => {
      const division = await DivisionModel.model.findOne({ where: { division_id: division_id } });
      const league = await LeagueModel.model.findOne({ where: { league_id: division.league_id} });
      if (!league) {
        logger.error("no league for divsion");
        return [];
      }

      return this.computeStanding(matches, league.league_id, league.team_size);
    });
  }
  async computeStanding(matches, league_id, team_size) {
    let scores = matches.reduce((accu, el) => {
      if (!accu[el.team_id]) {
        accu[el.team_id] = {};
        accu[el.team_id].team_id = el.team_id;
      }

      let outcome = el.outcome;
      if (!accu[el.team_id][outcome]) {
        accu[el.team_id][outcome] = 0;
      }
      if (!accu[el.team_id].score) {
        accu[el.team_id].score = 0;
      }

      accu[el.team_id][outcome] += 1;
      logger.debug("scores", el.scores);
      accu[el.team_id].score += (el.scores || [0]).reduce((a, s) => a+s);

      //logger.debug(el.team_id, JSON.stringify(accu[el.team_id], null, 2), el.match_id, el.outcome);
      return accu;
    }, {});

    const { teams, points_needed } = await Promise.props({
      teams: TeamModel.findAll2({
        where: {
          team_id: {
            [Sequelize.Op.in]: Object.keys(scores)
          }
        },
        include: [
          {
            model: UserModel.model,
            through: { attributes: [] }
          }
        ]
      }),
      points_needed: TournamentModel.findAll2({
        where: {
          league_id,
          state: {
            [Sequelize.Op.in]: ["created", "started"]
          }
        },
        order: ["start_dttm"],
        count: 1
      }).then(result => result && result[0] ? result[0].points_needed: Infinity)
    });

    return teams
      .filter(team => team.users.length === team_size)
      .map((team) => {
        let standing = {
          url: '/division/standing',
          win: scores[team.team_id].win || 0,
          loss: scores[team.team_id].loss || 0,
          draw: scores[team.team_id].draw || 0,
          points: calcPoints(scores[team.team_id]),
          team
        };
        standing.qualified = standing.points >= points_needed;
        return standing;
      })
      .sort((a, b) => b.points - a.points)
      .map((el, i, arr) => {
        el.points = Math.floor(el.points);
        if (i === 0) {
          el.rank = i + 1;
        } else {
          if (arr[i - 1].points == el.points) {
            el.rank = arr[i - 1].rank;
          } else {
            el.rank = i + 1;
          }
        }
        return el;
      });
  }
}

const singleton = new DivisionController();
module.exports = singleton;
