const UserController = require('../controllers/user');
const TeamController = require('../controllers/team');
const TarlyController = require('../classes/tarly_controller');
const sequelize = require('sequelize');
const error = require('../classes/error.js');
const logger = require('../api/logger');
const uuidv4 = require('uuid/v4');
const Analytics = require('../classes/analytics');

class InviteController {

  async getInviteUrl (viral_user_id, campaign_name, options) {
    let {team_id, division_id, league_id, campaign_content, campaign_medium, campaign_source, expire_dttm } = options;

    if (!campaign_medium) {
      campaign_medium = 'link';
    }

    if (!campaign_name) {
      throw new Error("missign campaign name");
    }

    const inviteModel = TarlyController.getModel('invite');

    let invite = await inviteModel.model.create(
      { invite_id: uuidv4(),
        campaign_name,
        campaign_source,
        campaign_medium,
        campaign_content,
        league_id,
        division_id,
        team_id,
        viral_user_id,
        expire_dttm});
    logger.debug(invite.get());
    return invite.get().invite_url;
  }

  async acceptInviteByCode(code, user_id) {
    const InviteModel = TarlyController.getModel('invite');

    let invite = await InviteModel.findOne({ code: code});
    if (!invite) {
      throw new error.NotFound(`Invite not found`);
    }

    return this.acceptInvite(invite, user_id);
  }

  async acceptTeamInvite(team_id, user_id) {
    if (!team_id) {
      throw new error.BadRequest("no team_id in team invite");
    }
    await TeamController.joinTeam(team_id, user_id);
  }

  async acceptLeagueInvite(league_id, user_id) {
    const UserLeagueModel = TarlyController.getModel('user_league');
    if (!league_id) {
      throw new error.BadRequest("no league_id in league invite");
    }
    try {
      await UserLeagueModel.create({
        owner_id: user_id,
        user_id: user_id,
        league_id: league_id
      });
      Analytics.track("user", "has_league", null, null, {league_id, user_id});
    } catch (err) {
      // silently fails on unique constraints cause it means user is already joined
      // the league/team
      if (!(err instanceof sequelize.UniqueConstraintError)) {
        throw new error.DefaultError(err);
      }
    }

    const UserModel = TarlyController.getModel('user');
    const LeagueModel = TarlyController.getModel('league');
    try{
      const league = await LeagueModel.findByPk(league_id);
      if(league.team_size === 1){
        const user = await UserModel.findByPk(user_id);
        const team = await TeamController.createAndJoinTeam({
          owner_id: user_id,
          team_id: uuidv4(),
          name: user.username,
          image_url : user.image_url,
          league_id: league_id,
          active_division_id: league.active_division_id
        }, user_id);
        await TeamController.emitTeamUsers(team.team_id);
      }
    }catch(err){
      throw new error.DefaultError(err);
    }

  }


  async acceptInvite(invite, user_id) {
    if (!invite) {
      throw new error.NotFound(`Invite not found`);
    }
    if (invite.team_id) {
      await this.acceptTeamInvite(invite.team_id, user_id);
      await TeamController.emitTeamUsers(invite.team_id);
    } else if (invite.league_id) {
      await this.acceptLeagueInvite(invite.league_id, user_id);
      await UserController.emit(user_id);
    } else {
      throw new error.BadRequest("No team or league in invite");
    }
    return invite;
  }

}
const singleton = new InviteController();
module.exports = singleton;
