const Sequelize = require('sequelize');
const moment = require('moment');
const error = require('../classes/error.js');
const logger = require('../api/logger');
const TC = require('../classes/tarly_controller');
const TarlySQLPool = require('../classes/sql_pool');
const UserController = require('./user.js');
const LeagueModel = TC.getModel('league');
const uuidv4 = require('uuid/v4');
const _ = require('lodash');
const Op = Sequelize.Op;

const ASYNC_REALTIME_UNIQUE_OPPONENT_QUERY = `
SELECT match.match_id
FROM match
       JOIN match_set ON (match_set.match_id = match.match_id)
       JOIN set ON (set.set_id = match_set.set_id)
       JOIN team ON (team.team_id = set.team_id)
       JOIN user_team ON (user_team.team_id = team.team_id)
WHERE match.division_id = :division_id
  AND match.state = 'started'
  AND match.created_dttm > :time_since
  AND match.deleted_dttm IS NULL
GROUP BY match.match_id
HAVING COUNT(DISTINCT set.set_id) = 1
   AND COUNT(DISTINCT user_team.user_id) = :team_size
   AND SUM(CASE WHEN user_team.user_id IN (:player_ids) THEN 1 ELSE 0 END) = 0
ORDER BY match.created_dttm ASC
LIMIT 1;
`;

const OTHER_MATCH_QUERY = `
SELECT old_challenger.match_id
FROM match
JOIN match_set AS opponent ON (match.match_id = opponent.match_id AND opponent.type = 'opponent')
JOIN match_set AS old_challenger on (opponent.set_id = old_challenger.set_id AND old_challenger.type = 'challenger')
WHERE match.match_id = (:match_id)
LIMIT 1;
`;

// const EXPIRED_MATCH_SET_QUERY = `
// SELECT set.*, match.match_id
// FROM match
// JOIN match_set ON match.match_id=match_set.match_id
// JOIN set ON match_set.set_id=set.set_id
// WHERE set.state='started'
//   AND match.deleted_dttm IS NULL
//   AND set.deleted_dttm IS NULL
//   AND NOW() > set.started_dttm+INTERVAL '30 minutes' * match.game_cnt
// ORDER by set.started_dttm+INTERVAL '30 minutes' * match.game_cnt asc
// LIMIT :count 
// `;

const EXPIRED_MATCH_SET_QUERY = `
 SELECT
    set.set_id,
    match.match_id
FROM
    set
    JOIN match_set ON match_set.set_id = set.set_id
    JOIN match ON match_set.match_id = match.match_id
WHERE
    match.deleted_dttm IS NULL
    AND set.deleted_dttm IS NULL
    AND set.state != 'ended'
LIMIT :count 
`;





class MatchController {
  constructor() {
    this.matchModel = TC.getModel('match');
    this.setModel = TC.getModel('set');
    this.userModel = TC.getModel('user');
    this.teamModel = TC.getModel('team');
    this.matchSetModel = TC.getModel('match_set');
    this.setUserModel = TC.getModel('set_user');
    this.tournamentModel = TC.getModel('tournament');
  }

  calculateTournamentOrderKey(match) {
    // only 32 bit
    // max seed = 4096

    let val = match.tournament_round * match.tournament_score;
    val = val * 10000;
    val += 10000;
    val -= match.tournament_seed;
    // val = val << 8;
    // val += parseInt(match.match_id.split("-")[2], 16);
    return val;

  }

  async createTournamentMatch(tournament_id, team_ids, game_cnt, tournament_type, tournament_round, tournament_bracket, tournament_seed, bracket_state) {

    const tournament = await this.tournamentModel.findById(tournament_id);
    if (!tournament || !tournament.league_id) {
      throw new error.BadRequest(`Invalid tournament ${tournament_id}`);
    }
    const league_id = tournament.league_id;
    const league_game_id = await this.getLeagueGameId(league_id);

    let obj = {
      match_id: uuidv4(),
      owner_id: 'service',
      tournament_id,
      tournament_round,
      tournament_type, 
      tournament_bracket,
      tournament_seed,
      tournament_score: 0,
      league_id,
      state: 'created',
      game_id: league_game_id
    };

    obj.tournament_order = this.calculateTournamentOrderKey(obj);

    if (bracket_state === 'bye' || bracket_state === 'eliminated') {
      obj.state = 'ended';
    }
    if (game_cnt) {
      obj.game_cnt = game_cnt;
    }
    const match = await this.matchModel.model.create(obj, { raw: false });

    for (let team_id of team_ids) {
      const team = await this.teamModel.findById(team_id);

      let challenger_ids = new Set(team.users.map(u => u.user_id));
      challenger_ids = Array.from(challenger_ids);


      const setObj = {
        set_id: uuidv4(),
        tournament_id,
        owner_id: 'service',
        team_id,
        state: 'created'
      };
      if (obj.state === 'ended') {
        setObj.outcome = bracket_state;
        setObj.state = 'ended';
        setObj.scores = [];
        if (game_cnt && game_cnt !== -1) {
          while (setObj.scores.length < game_cnt) {
            setObj.scores.push(0);
          }
          setObj.round = game_cnt + 1;
        }
      }

      const set = await this.setModel.model.create(setObj, { raw: false });

      await this.matchSetModel.model.create(
        {
          owner_id: 'service',
          match_set_id: uuidv4(),
          set_id: set.set_id,
          match_id: match.match_id
        },
        { raw: false }
      );

      for (const user_id of challenger_ids) {
        await this.setUserModel.model.create(
          { owner_id: user_id, set_user_id: uuidv4(), set_id: set.set_id, user_id },
          { raw: false }
        );
        if (obj.state !== 'ended') {
          await this.userModel.updateById(user_id, { active_match_id: match.match_id });
          await UserController.emit(user_id);
        }
      }
    }

    return match.match_id;
  }

  async getLeagueGameId(league_id){
    const league = await LeagueModel.model.findByPk(league_id);
    return league.game_id;
  }

  async createMatch(user_id, team_id, game_cnt) {
    const team = await this.teamModel.findById(team_id);
    if(!team){
      throw new error.BadRequest(`Team ${team_id} does not exist`);
    }
    const division_id = team.active_division_id;
    const league_id = team.league_id;

    if (!division_id) {
      throw new error.BadRequest(`Team ${team_id} is missing division`);
    }

    if (!league_id) {
      throw new error.BadRequest(`Team ${team_id} is missing league_id`);
    }

    const league_game_id = await this.getLeagueGameId(league_id);

    if (!team.users.some(u => u.user_id === user_id)) {
      throw new error.BadRequest('User not in this team');
    }


    let challenger_ids = new Set(team.users.map(u => u.user_id));
    challenger_ids = Array.from(challenger_ids);
    // fixme - must use useMaster
    // This is not atomic...
    for (const user_id of challenger_ids) {
      let user = await this.userModel.findByPk(user_id);
      if (!user) {
        throw new error.BadRequest('invalid user_id ' + user_id);
      }
      if (user.active_match_id != null) {
        logger.error('User already in match', user.user_id, user.username);
        throw new error.Conflict('user already in match');
      }
    }
    let obj = { league_id, game_id: league_game_id, division_id, match_id: uuidv4(), owner_id: user_id, state: 'created' };
    if (game_cnt) {
      obj.game_cnt = game_cnt;
    }

    const match = await this.matchModel.model.create(obj, { raw: false });

    for (const user_id of challenger_ids) {
      await this.userModel.updateById(user_id, { active_match_id: match.match_id });
    }

    const set = await this.setModel.model.create(
      { set_id: uuidv4(), division_id, owner_id: user_id, team_id, state: 'created' },
      { raw: false }
    );

    await this.matchSetModel.model.create(
      {
        type: 'challenger', // TODO: delete this
        owner_id: user_id,
        match_set_id: uuidv4(),
        set_id: set.set_id,
        match_id: match.match_id
      },
      { raw: false }
    );

    for (const user_id of challenger_ids) {
      await this.setUserModel.model.create(
        { owner_id: user_id, set_user_id: uuidv4(), set_id: set.set_id, user_id },
        { raw: false }
      );
      await UserController.emit(user_id);
    }

    return match.match_id;
  }

  async reserveMatch(match) {
    const league = await TC.getModel('league').findById(match.sets[0].team.league_id);

    const match_id = await TarlySQLPool.transaction(async transaction => {
      const oldMatches = await TarlySQLPool.query(ASYNC_REALTIME_UNIQUE_OPPONENT_QUERY, {
        type: Sequelize.QueryTypes.SELECT,
        replacements: {
          division_id: match.division_id,
          team_size: league.team_size,
          player_ids: match.sets[0].team.users.map(u => u.user_id),
          time_since: moment()
            .tz(league.timezone)
            .startOf('month')
            .toISOString()
        },
        transaction,
        lock: transaction.LOCK.UPDATE
      });

      if (!oldMatches || !oldMatches.length) {
        logger.warn('No qualified matches at all', match);

        // start new match without opponent

        // await this.matchModel.model.update(
        //   {
        //     state: 'started'
        //   },
        //   {
        //     where: { match_id: match.match_id },
        //     transaction
        //   }
        // );

        // start new set
        // await this.setModel.model.update(
        //   {
        //     state: 'started',
        //     round: 1,
        //     started_dttm: Date.now()
        //   },
        //   {
        //     where: { set_id: match.sets[0].set_id },
        //     transaction
        //   }
        // );

        return match.match_id;
      }

      const oldMatch = await this.matchModel.model.findByPk(oldMatches[0].match_id, {
        useMaster: true,
        include: this.matchModel.config.include
      });

      // associate new set to the old match
      await TC.getModel('match_set').model.create(
        {
          type: 'opponent', // TODO: delete this
          owner_id: match.owner_id,
          match_set_id: uuidv4(),
          set_id: match.sets[0].set_id,
          match_id: oldMatch.match_id
        },
        { transaction }
      );

      // start the new set
      await this.setModel.model.update(
        {
          state: 'started',
          started_dttm: Date.now()
        },
        {
          where: { set_id: match.sets[0].set_id },
          transaction
        }
      );

      await this.matchModel.model.update(
        {
          data: { state: Object.assign(oldMatch.data.state, match.data.state) }
        },
        {
          where: { match_id: oldMatch.match_id },
          transaction
        }
      );

      const [nUpdated] = await this.userModel.model.update(
        { active_match_id: oldMatch.match_id },
        {
          where: {
            active_match_id: match.match_id
          },
          transaction
        }
      );

      if (nUpdated !== league.team_size) {
        logger.error('Wrong number of updated users ???');
      }

      await this.matchModel.deleteByIdAndReturn(match.match_id);

      return oldMatch.match_id;
    });

    const newMatch = await this.matchModel.model.findByPk(match_id, {
      useMaster: true,
      include: this.matchModel.config.include
    });

    for (const set of newMatch.sets) {
      for (let user of set.team.users) {
        await UserController.emit(user.user_id);
      }
    }

    return newMatch;
  }

  async endMatch(match, cancel, set_id) {
    // deprecated - just here for use during deploye
    const where = { active_match_id: match.match_id };

    if (set_id) {
      const set = match.sets.find(set => set.set_id === set_id);
      where.user_id = {
        $in: set.team.users.map(u => u.user_id)
      };
    }

    let users = await this.userModel.updateWhere({ active_match_id: null }, where);

    for (let u of users) {
      await UserController.emit(u.user_id);
    }

    let newMatch = null;
    if (cancel) {
      newMatch = await this.matchModel.deleteByIdAndReturn(match.match_id);
    } else {
      newMatch = this.matchModel.findById(match.match_id);
    }
    return newMatch;
  }

  // get match only if user is in it
  async getOwnMatch(match_id, user_id) {
    const opts = {
      useMaster: true,
      paranoid: false, // include deleted
      plain: true, // only one
      include: _.cloneDeep(this.matchModel.config.include),
      where: { match_id: match_id }
    };

    opts.include.push({
      required: true,
      model: this.setModel.model,
      as: 'owner',
      through: { attributes: [] },
      include: [
        {
          required: true,
          model: this.teamModel.model,
          as: 'team',
          include: [
            {
              required: true,
              model: this.userModel.model,
              where: { user_id }, // find with user
              through: { attributes: [] }
            }
          ]
        }
      ]
    });

    return this.matchModel.findAll2(opts);
  }

  async get(match_id) {
    const opts = {
      useMaster: true,
      paranoid: false, // include deleted
      plain: true, // only one
      include: _.cloneDeep(this.matchModel.config.include),
      where: { match_id: match_id }
    };

    return this.matchModel.findAll2(opts);
  }

  async getOtherMatch(match_id) {
    const match = await TarlySQLPool.query(OTHER_MATCH_QUERY, {
      type: Sequelize.QueryTypes.SELECT,
      replacements: {
        match_id
      }
    });
    return match[0];
  }

  async getMatchesForWeek(team_id) {
    const team = await TC.getModel('team').findById(team_id);
    const league = await TC.getModel('league').findById(team.league_id);

    let startOfWeek = moment()
      .tz(league.timezone)
      .startOf('isoWeek');

    const startOfMonth = moment()
      .tz(league.timezone)
      .startOf('month');

    let gte;
    if (startOfMonth.isAfter(startOfWeek)) {
      gte = startOfMonth.toISOString();
    } else {
      gte = startOfWeek.toISOString();
    }
    logger.debug('gte', gte);

    let matches = await this.matchModel.model.findAll({
      where: {
        tournament_id: {
          [Op.eq]: null
        }
      },
      include: this.matchModel.config.include.concat([
        {
          required: true,
          model: this.setModel.model,
          as: 'owner',
          where: {
            created_dttm: {
              $gte: gte
            }
          },
          through: { attributes: [] },
          include: [
            {
              required: true,
              model: this.teamModel.model,
              as: 'team',
              where: { team_id }
            }
          ]
        }
      ]),
      order: [[{ model: this.setModel.model, as: 'owner' }, 'created_dttm', 'asc']],
      useMaster: true
    });

    matches.forEach(m => {
      delete m.dataValues.owner;
    });

    return matches;
  }
  async getExpired(count = 100) {

    let sets = await TarlySQLPool.query(EXPIRED_MATCH_SET_QUERY,
      {
        type: Sequelize.QueryTypes.SELECT,
        replacements: { count },
      });
    return sets;
  }
}

const singleton = new MatchController();

module.exports = singleton;
