const UserController = require('../controllers/user');
const TarlyController = require('../classes/tarly_controller');
const sequelize = require('sequelize');
const error = require('../classes/error.js');
const Analytics = require('../classes/analytics');
const User = require('../api/user');
const Notification = require('../api/notification');
const TarlySQLPool = require('../classes/sql_pool');

const UserTeamModel = TarlyController.getModel('user_team');
const TeamModel = TarlyController.getModel('team');
const LeagueModel = TarlyController.getModel('league');

const TEAM_MEDAL_RECALC = `
WITH medals AS (
  SELECT
      set.team_id AS team_id,
      COUNT(CASE WHEN set.outcome='win' THEN 1 END) as gold_cnt,
      COUNT(CASE WHEN set.outcome='loss' OR set.outcome='draw' THEN 1 END) as silver_cnt
  FROM
      tournament
      JOIN match ON match.tournament_id = tournament.tournament_id
          AND match.tournament_bracket = 'upper'
          AND tournament.round_cnt = match.tournament_round
      JOIN match_set ON match_set.match_id = match.match_id 
      JOIN set ON set.set_id = match_set.set_id
  GROUP BY set.team_id
)
UPDATE team
SET gold_cnt=medals.gold_cnt,
    silver_cnt=medals.silver_cnt
FROM medals
WHERE team.team_id=medals.team_id;
`;

class TeamController {

  // TODO bring this to leaguecontroller
  async joinLeague(league_id, user_id) {
    const UserLeagueModel = TarlyController.getModel('user_league');
    if (!league_id) {
      throw new error.BadRequest("no league_id in team join league");
    }

    try {
      await UserLeagueModel.create({
        owner_id: user_id,
        user_id: user_id,
        league_id: league_id
      });
      Analytics.track("user", "has_league", null, null, {league_id, user_id});
    } catch (err) {
      // silently fails on unique constraints cause it means user is already joined
      // the league/team
      if (!(err instanceof sequelize.UniqueConstraintError)) {
        throw new error.DefaultError(err);
      }
    }
  }

  async createUserTeam(team_id, league_id, user_id) {
    try {
      await UserTeamModel.create({
        owner_id: user_id,
        user_id: user_id,
        league_id: league_id,
        team_id: team_id
      });
    } catch (err) {
      if (!(err instanceof sequelize.UniqueConstraintError)) {
        throw new error.DefaultError(err);
      } else {
        throw new error.Conflict(`You're already in a team`);
      }
    }
  }

  async createAndJoinTeam(create_options, user_id) {
    const team = await TeamModel.model.create(
      create_options,
      { raw: false });
    Analytics.track("user", "create", "team", null, {team_id: team.team_id, user_id});

    await this.joinTeam(team.team_id, user_id);
    return team;
  }
  async recalcTeamMedals() {
    await TarlySQLPool.query(TEAM_MEDAL_RECALC, { });
  }

  async joinTeam(team_id, user_id) {
    const team = await TeamModel.findByPk(team_id, { useMaster: true });
    if (!team) {
      throw new error.NotFound(`Team does not exist`);
    }

    const league = await LeagueModel.findByPk(team.league_id, { useMaster: true });
    if (!league) {
      throw new error.NotFound(`League does not exist`);
    }

    const old_teammates = await UserTeamModel.findAll2({
      where: { team_id },
      useMaster: true
    });

    if (old_teammates.length === league.team_size) {
      throw new error.Forbidden('The team is full');
    }

    await this.joinLeague(team.league_id, user_id);
    await this.createUserTeam(team_id, team.league_id, user_id);

    Analytics.track("user", "join", "team", null, {team_id: team.team_id, user_id});
    // old_teammates + new joined user (1)
    if (old_teammates.length + 1 === league.team_size) {

      Analytics.track("user", "has_team", null, null, {team_id, user_id});
      let user = await User.get(user_id);
      for (let old of old_teammates) {
        Analytics.track("user", "has_team", null, null, {team_id, user_id: old.user_id});
        let notification = {
          "title": "You are ready to play!",
          "message": `${user.username} has accepted your team request`,
          "icon_url": team.image_url
        };
        Notification.send(old.user_id, notification, {campaign_name: "you_are_ready_to_play"});
      }
    }
  }

  async emitTeamUsers(team_id) {
    const opts = {
      where: { team_id },
      useMaster: true,
      paranoid:false
    };
    const teammates = await UserTeamModel.findAll2(opts);
    for (const teammate of teammates) {
      UserController.emit(teammate.user_id);
    }
  }
}
const singleton = new TeamController();
module.exports = singleton;
