const generateCode = require('password-generator');

const moment = require('moment');
const logger = require('../api/logger');
const error = require('../classes/error');
const TarlyController = require('../classes/tarly_controller');

const BotCommandModel = require('../models/bot_commands');
const SceneShareModel = require('../models/scene_share');

const Promise = require('bluebird');

const getTournamentById = tournament_id => {
  return TarlyController.getModel('tournament')
    .findById(tournament_id)
    .then(tournament => {
      if (!tournament) {
        return Promise.reject(new error.NotFound('tournament does not exist'));
      }
      return tournament;
    });
};

class TournamentTeamController {
  constructor() {
    this.create = this.create.bind(this);
    this.update = this.update.bind(this);
    this.delete = this.delete.bind(this);
    this.join = this.join.bind(this);
    this.leave = this.leave.bind(this);
  }
  create({ tournament_id, name, stream_id, username }, tries = 0) {
    return getTournamentById(tournament_id).then(tournament => {
      if (
        moment(tournament.start_dttm)
          .add(1, 'h')
          .isBefore(moment())
      ) {
        return Promise.reject(
          new error.Forbidden('cannot create a team more than 1h after the start of a tournament')
        );
      }
      const code = generateCode(6, false, /\d/);
      if (!name) {
        name = `${username}'s team`;
      }
      const model = TarlyController.getModel('tournament_team');
      return model
        .create({
          tournament_id,
          name,
          stream_id,
          usernames: [username],
          player_stream_ids: [stream_id],
          code
        })
        .catch(err => {
          logger.error('failed to create tournament team', err);
          //retry up to 3 times, error is likely a conflict on the code since it's not guaranteed to be unique
          //collisions are mathematically highly unlikely but possible
          if (tries < 3) {
            return this.create({ tournament_id, name, stream_id, username }, tries + 1);
          }
          return Promise.reject(err);
        })
        .then(team => this.runTeamJoinActions(tournament, stream_id).then(() => team));
    });
  }

  update({ id, name, stream_id, usernames, player_stream_ids }) {
    const model = TarlyController.getModel('tournament_team');
    return model.findById(id).then(team => {
      //first check if we're the owner of the team, otherwise we're not allowed to edit at all
      if (team.stream_id !== stream_id) {
        return Promise.reject(
          new error.Forbidden('you are not the owner of the team you are trying to edit')
        );
      }
      return model.updateById(id, Object.assign({}, team, { usernames, player_stream_ids, name }));
    });
  }

  delete({ id, stream_id }) {
    const model = TarlyController.getModel('tournament_team');
    return model.findById(id).then(team => {
      //first check if we're the owner of the team, otherwise we're not allowed to delete at all
      if (team.stream_id !== stream_id) {
        return Promise.reject(
          new error.Forbidden('you are not the owner of the team you are trying to edit')
        );
      }
      //check if the tournament is already running for > 1h if so we cannot delete
      return getTournamentById(team.tournament_id).then(tournament => {
        if (
          moment(tournament.start_dttm)
            .add(1, 'h')
            .isBefore(moment())
        ) {
          return Promise.reject(
            new error.Forbidden(
              'cannot disband a team more than 1h after the start of a tournament'
            )
          );
        }
        return model.deleteById(id);
      });
    });
  }

  join({ code, username, stream_id }) {
    const model = TarlyController.getModel('tournament_team');
    //check if team for code exists
    return model.findOne({ code })
      .then(team => {
        if (!team) {
          return Promise.reject(new error.NotFound('team for code does not exist'));
        }
        //prevent double joining a tournament
        if (team.usernames.indexOf(username) > -1 || team.player_stream_ids.indexOf(stream_id) > -1) {
          return Promise.resolve(team);
        }
        return getTournamentById(team.tournament_id)
          .then(tournament => {
            const size = tournament.team_size;
            if (team.usernames.length === size) {
              return Promise.reject(new error.Conflict('team is full'));
            }
            const usernames = team.usernames.concat(username);
            const player_stream_ids = team.player_stream_ids.concat(stream_id);
            return Promise.props({
              team: model.updateById(team.id, { usernames, player_stream_ids }),
              tournament
            });
          })
          .then(({ team, tournament }) => this.runTeamJoinActions(tournament, stream_id).then(() => team));
    });
  }

  leave({ id, username, stream_id }) {
    const model = TarlyController.getModel('tournament_team');
    //check if team for code exists
    return model.findById(id).then(team => {
      if (!team) {
        return Promise.reject(new error.NotFound('team does not exist'));
      }
      if (stream_id === team.stream_id) {
        //the team creator is leaving, this is actually === a disband of the team
        return this.delete({ id, stream_id });
      }
      const usernames = team.usernames.filter(name => name !== username);
      const player_stream_ids = team.player_stream_ids.filter(s_id => s_id !== stream_id);
      return model.updateById(team.id, { usernames, player_stream_ids });
    });
  }

  runTeamJoinActions(tournament, stream_id) {
    const proms = [
      SceneShareModel.copyScenes(stream_id, tournament.scene_ids)
    ];
    const bot_command_message = `I'm playing in the ${tournament.name} right now. Rules, format, & more here: https://bebo.com/t/${tournament.slug}`;
    const bot_command_keywords = ['bebo', 'tournament', 'rules', 'score', 'tourney', 'leaderboard', 'format'];
    bot_command_keywords.forEach(keyword => {
      const cmd = {
        keyword,
        message: bot_command_message,
        active: true,
        is_help: false,
        stream_id
      };
      proms.push(BotCommandModel.doUpsert(cmd));
    });
    return Promise.all(proms);
  }
}

const singleton = new TournamentTeamController();
module.exports = singleton;
