const bcrypt = require('bcryptjs');
const uuidv4 = require('uuid/v4');
const gamertag = require('../classes/gamertag');
const sequelize = require('sequelize');

const Analytics = require('../classes/analytics');
const Broadcast = require('../api/broadcast');
const logger = require('../api/logger');

const config = require('../config.js');

const TarlyController = require('../classes/tarly_controller');
const SMS = require('../classes/sms');
const cache = require('bebo-node-commons').RedisCache;
const error = require('../classes/error.js');

const UserModel = TarlyController.getModel('user');
const TeamModel = TarlyController.getModel('team');
const LeagueModel = TarlyController.getModel('league');
const AccessTokenModel = TarlyController.getModel('access_token');

const Redis = cache.getWriteClient(config.PHONE_AUTH_DB);

const AUTH_CODE_TTL = 5 * 60; // 5 minutes
const PIN_SMS_TEMPLATE = 'Bebo login pin: ';
const SALT_ROUNDS = 12;


const googleLibPhoneNumber = require('google-libphonenumber');
const phoneUtil = googleLibPhoneNumber.PhoneNumberUtil.getInstance();
const PNF = googleLibPhoneNumber.PhoneNumberFormat;

function replaceBetween(str, replace_with, start, end) {
  return str.substr(0, start)
    .concat(str.substr(start, end).replace(/./g, replace_with))
    .concat(str.substr(end));
}

class UserController {

  async findByPk(user_id, options) {
    if (!options) {
      options = {};
    }
    let opts = Object.assign({
      paranoid: true,
      useMaster: true,
      raw: false,
      plain: true,
      include: []
    }, options);
    opts.include.push({
      model: LeagueModel.model,
      through: { attributes: [] },
      include: [{
        model: TeamModel.model,
        through: {
          attributes: [],
          where: {user_id: user_id},
        },
        include: {
          model: UserModel.model,
          through: { attributes: [] },
          attributes: ['username', 'user_id', 'image_url'],
        }
      }]});

    return UserModel.model.findByPk(user_id, opts);
  }

  async emit(user_id) {
    let entity = await this.findByPk(user_id);
    if (! entity) {
      return;
    }
    let payload = UserModel.entityToPayload(entity);
    let url = payload.url;
    let top_id_key = UserModel.getTopLevelIdKey();
    let to_id = payload[top_id_key];

    // the id must be the top level id - not the id of the element
    return Broadcast.emit(url, to_id, payload)
      .then(data => {
        logger.debug('Broadcast.emit success', payload, data, url, to_id);
      })
      .catch(err => {
        logger.error('Broadcast.emit', err, url, to_id);
      });
  }


  async signupWithPhoneNumber(phone_number, pin, campaign, platform) {
    const username = await gamertag();

    const key = phone_number + '_' + pin;
    let json = await Redis.getAsync(key);
    if (json == null) {
      throw new error.Unauthorized('wrong phone_number or pin');
    }
    let data = JSON.parse(json);
    campaign = Object.assign(campaign, data);

    let user = {
      username,
      phone_number,
      campaign,
    };

    if (platform === "pc") {
      user.has_pc = true;
    }

    user = await this.createUser(user);

    if (platform != null) {
      Analytics.track("user", "signup", platform, null, { user });
    }
    Analytics.track("user", "signup", null, null, { user });

    return user;
  }

  async signupWithPassword(username, password, phone_number, email, campaign, platform) {

    username = this.normalizeUsername(username);

    const hashedPassword = await bcrypt.hash(password, SALT_ROUNDS);
    let user = {
      username,
      password: hashedPassword,
      campaign,
      email: email || undefined,
      phone_number: phone_number || undefined
    };

    user = await this.createUser(user);
    if (platform != null) {
      Analytics.track("user", "signup", platform, null, { user });
    }
    Analytics.track("user", "signup", null, null, { user });
    return user;
  }

  async loginWithPhoneNumber(phone_number, pin, platform) {

    const key = phone_number + '_' + pin;
    let json = await Redis.getAsync(key);
    if (json == null) {
      throw new error.Unauthorized('wrong phone_number or pin');
    }

    let user = await UserModel.findOne({ phone_number }, false, false);
    if (!user) {
      throw new error.NotFound('User Not Found');
    }

    await this.savePlatform(user, platform);

    user = await this.findByPk(user.user_id);
    user = user.get();
    let access_token = await this.generateAccessToken(user.user_id);
    user.access_token = access_token;
    this.trackPlatformOnLogin(user, platform);
    return user;
  }


  async loginWithPassword(username, password, platform) {
    username = this.normalizeUsername(username);

    let user = await UserModel.findOne({ username }, false, true);
    if (!user) {
      throw new error.NotFound('User Not Found');
    }

    let ok = await bcrypt.compare(password, user.password);
    if (! ok) {
      throw new error.Unauthorized('wrong password or username');
    }

    await this.savePlatform(user, platform);

    user = await this.findByPk(user.user_id);
    user = user.get();
    let access_token = await this.generateAccessToken(user.user_id);
    user.access_token = access_token;

    this.trackPlatformOnLogin(user, platform);
    return user;
  }

  async savePlatform(user, platform) {
    if (platform === "pc") {
      await UserModel.updateById(user.user_id, {has_pc: true}, false);
    }
  }

  async trackPlatformOnLogin(user, platform) {
    if (platform != null) {
      Analytics.track("user", "login", platform, null, { user });
    }
    Analytics.track("user", "login", null, null, { user });
  }

  async createUser(userObj) {
    try {
      let user = await UserModel.create(userObj);
      user = await this.findByPk(user.user_id);
      user = user.get();
      let access_token = await this.generateAccessToken(user.user_id);
      user.access_token = access_token;
      return user;
    } catch(err) {
      if (err instanceof sequelize.UniqueConstraintError) {
        logger.info('unique conflict at user signup', err);
        throw new error.Conflict('username, email or phone_number taken');
      }
      throw err;
    }
  }

  normalizeUsername(username) {
    username = username.toLowerCase();
    if (!/^[a-z0-9_\-.]{3,}$/.test(username)) {
      throw new error.BadRequest('invalid username');
    }
    return username;
  }

  randomPin() {
    return  `${Math.floor(Math.random() * 1000000)}`.padStart(6, "0");
  }

  formatPinMessage(phoneNumber) {
    let phoneObj = phoneUtil.parseAndKeepRawInput(phoneNumber);
    const phone_number_localize = 
      phoneUtil.format(phoneObj, PNF.NATIONAL);
    const hidden_phone_number = replaceBetween(
      phone_number_localize, '*', 0, 
      phone_number_localize.length - 4);
    return "Pin sent to " + hidden_phone_number;
  }

  async sendValidationSMS(phone_number, data) {

    const pin = this.randomPin();
    const key = phone_number + '_' + pin;
    const json = JSON.stringify(data);

    await Redis.setAsync(key, json, 'EX', AUTH_CODE_TTL);
    await SMS.sendMessage(phone_number, PIN_SMS_TEMPLATE + pin);
  }

  async generateAccessToken(user_id) {

    const access_token = await AccessTokenModel.create({
      access_token: uuidv4(),
      owner_id: user_id,
      user_id
    });

    return access_token.access_token;
  }

}
const singleton = new UserController();
module.exports = singleton;
