const AWS = require('aws-sdk');
const request = require('request-promise');
const streamToArray = require('stream-to-array');
const uuid = require('uuid');
const zlib = require('zlib');

const config = require('../config.js');
const error = require('../classes/error.js');
const logger = require('../api/logger');

AWS.config.setPromisesDependency(require('bluebird'));
AWS.config.accessKeyId = config.aws_access_key;
AWS.config.secretAccessKey = config.aws_secret;
var s3 = new AWS.S3();


const CrashPad = {
  route: '/breakpad',

  post: function(req, res, next) {
    if (!req.busboy) {
      return next(new error.BadRequest());
    }

    let props = {
      version: '0.0.0',
      product: 'unknown',
      crashdump_id: uuid.v4(),
      created_dttm: new Date()
    };
    let dumpFile = null;
    let streamOps = [];

    streamOps.push(
      new Promise(resolve => {
        req.busboy.on('file', (fieldname, file, filename) => {
          props.crashdump_id = filename.replace(/.dmp/i, '');
          streamToArray(file)
            .then(parts => {
              let buffers = [];
              for (let i = 0; i < parts.length; i++) {
                let part = parts[i];
                if (part instanceof Buffer) {
                  buffers.push(part);
                } else {
                  buffers.push(new Buffer(part));
                }
              }
              return Buffer.concat(buffers);
            })
            .then(buffer => {
              dumpFile = buffer;
              resolve();
            })
            .catch(err => {
              logger.error('error in file processing', err);
            });
        });
      })
    );

    req.busboy.on('field', (fieldname, val, fieldnameTruncated, valTruncated) => {
      if (fieldname === 'prod') {
        props['product'] = val;
      } else if (fieldname === 'ver') {
        props['version'] = val;
      } else if (fieldname) {
        props[fieldname] = val.toString();
      }
    });

    props.ip = req.ip;
    props.headers = req.headers;
    req.busboy.on('error', err => {
      logger.error('ERROR', err);
    });

    req.busboy.on('finish', () => {
      Promise.all(streamOps)
        .then(() => {
          let key = `breakpad/${props.product}/${props.version}/${props.crashdump_id}`;
          props.url = `https://s3-us-west-1.amazonaws.com/bebo-app/${key}.dmp`;
          let jsonUrl = `https://s3-us-west-1.amazonaws.com/bebo-app/${key}.json`;
          let params = { Bucket: 'bebo-app', Key: key + '.dmp', Body: dumpFile };
          return s3
            .putObject(params)
            .promise()
            .then(() => {
              let json = JSON.stringify(props);
              let params = { Bucket: 'bebo-app', Key: key + '.json', Body: json };
              return s3.putObject(params).promise();
            })
            .then(() => {
              const opts = {
                method: 'POST',
                url:
                  '***REMOVED***',
                json: true,
                body: {
                  text: `New crash ${props.product} ${props.version} ${jsonUrl}`
                }
              };
              return request(opts);
            })
        })
        .then(() => {
          res.code = 200;
          return next();
        })
        .catch(err => {
          return next(new error.DefaultError(err));
        });
    });
    let stream = req;
    switch (req.headers['content-encoding'] || 'identity') {
      case 'gzip':
        stream = req.pipe(zlib.createGunzip());
        break;
      case 'deflate':
        stream = req.pipe(zlib.createInflate());
        break;
      case 'identity':
        break;
      default:
        return next(new error.UnsupportedMediaType('content-encoding not supported'));
    }
    stream.pipe(req.busboy);
  }
};

module.exports = CrashPad;
