
const error = require('../classes/error');
const logger = require('../api/logger');
const TarlyController = require('../classes/tarly_controller');
const asyncMiddleware = require('../classes/async-middleware');
const MatchController = require('../controllers/match');
const AuthenticatedUser = require('../middleware/authenticated_user');
const _ = require('lodash');

module.exports = {
  route: '/match',
  wrapper: AuthenticatedUser,
  post: asyncMiddleware(async function(req, res, next) {

    const matchModel = TarlyController.getModel('match');
    const team_id = req.body.team_id;
    const team_ids = req.body.team_ids;
    const tournament_id = req.body.tournament_id;
    const tournament_round = req.body.tournament_round;
    const tournament_seed = req.body.tournament_seed;
    const tournament_bracket = req.body.tournament_bracket;
    const tournament_type = req.body.tournament_type;
    const game_cnt = req.body.game_cnt;
    const bracket_state = req.body.state;

    let match_id;
    if (tournament_id) {
      if (! req.is_admin) {
        return next(new error.Forbidden());
      }
      match_id = await MatchController.createTournamentMatch(tournament_id, team_ids, game_cnt, tournament_type, tournament_round, tournament_bracket, tournament_seed, bracket_state);
    } else {
      const user_id = req.acting_user.user_id;
      match_id = await MatchController.createMatch(user_id, team_id, game_cnt);
    }

    const match = await matchModel.model.findByPk(match_id, {
      useMaster: true,
      include: matchModel.config.include,
      paranoid: false,
      raw: false
    });

    res.result = [match];
    return next();
  }),

  put: asyncMiddleware(async function(req, res, next) {

    if (! req.is_admin) {
      return next(new error.Forbidden());
    }

    const match_id = req.body.match_id;
    const matchModel = TarlyController.getModel('match');

    // TODO update can return result - better to do that
    let match = await matchModel.findByPk(match_id);
    if (match == null) {
      throw new error.NotFound();
    }
    match = match.get();
    match = Object.assign(match, req.body);

    req.body.tournament_order = MatchController.calculateTournamentOrderKey(match);
    await matchModel.updateById(match_id, req.body);

    match = await matchModel.model.findByPk(match_id, {
      include: matchModel.config.include,
      paranoid: false,
      useMaster: true,
      raw: false
    });

    res.result = [match];
    return next();
  }),

  get: asyncMiddleware(async function(req, res, next) {
    const matchModel = TarlyController.getModel('match');
    const setModel = TarlyController.getModel('set');
    const userTeamModel = TarlyController.getModel('user_team');

    let opts = _.cloneDeep(req.sqlOptions);
    opts.include = _.cloneDeep(matchModel.config.include);
    if (req.is_admin) {
      opts.useMaster = true;
    }
    let where = opts.where;
    if (!where) {
      throw new error.BadRequest('no where, cant get all');
    }
    let team_id = where.team_id;
    let user_id = where.user_id;
    delete where.team_id;
    delete where.user_id;
    if(user_id) {
      const teams = await userTeamModel.findAll2({where: {user_id}});
      const team_ids = teams.map(t => t.team_id);
      opts.include.push({
        required: true,
        model: setModel.model,
        where: {team_id: {$in: team_ids}},
        as: 'owner',
        through: { attributes: [] },
      });
      opts.order = [[{ model: setModel.model, as: 'owner' }, 'created_dttm', 'desc']];
    }else if (team_id) {
      opts.include.push({
        required: true,
        model: setModel.model,
        where: {team_id},
        as: 'owner',
        through: { attributes: [] },
      });
      opts.order = [[{ model: setModel.model, as: 'owner' }, 'created_dttm', 'desc']];
    }
    let matches = await matchModel.findAll2(opts);
    matches.forEach(m => {
      delete m.dataValues.owner;
    });

    res.result = matches;
    return next();
  })
};
