const Sequelize = require('sequelize');
const error = require('../classes/error');
const AuthenticatedUser = require('../middleware/authenticated_user');
const TarlyController = require('../classes/tarly_controller');
const asyncMiddleware = require('../classes/async-middleware');
const logger = require('../api/logger');

module.exports = {
  route: '/set/view',
  wrapper: AuthenticatedUser,
  post: asyncMiddleware(async function(req, res, next) {
    const owner_id = req.acting_user.user_id;
    const set_id = req.body.set_id;
    const user_id = req.body.user_id;

    const setViewModel = TarlyController.getModel('set_view');
    const userModel = TarlyController.getModel('user');
    const matchModel = TarlyController.getModel('match');
    const setModel = TarlyController.getModel('set');

    let setView;
    try {
      setView = await setViewModel.create({ user_id, set_id, owner_id});
    } catch (err) {
      if (err instanceof Sequelize.UniqueConstraintError) {
        logger.debug("user has already viewed");
        return next();
      }
      return next(new error.DefaultError(err));
    }

    const match = await matchModel.model.findAll({
      include: [
        {
          required: true,
          model: setModel.model,
          as: 'owner',
          where: { set_id },
          through: { attributes: [] }
        },
        {
          model: setModel.model,
          as: 'sets',
          through: { attributes: [] },
          include: [
            {
              model: setViewModel.model,
              include: [
                {
                  model: userModel.model,
                  attributes: ['username']
                }
              ]
            }
          ]
        }
      ],
      plain: true,
      useMaster: true
    });

    if (!match) {
      throw new error.BadRequest('cannot find match');
    }

    let viewers = [
      ...new Set([].concat(...match.sets.map(set => set.set_views.map(sv => sv.user.username))))
    ];

    let viewer_names =
      viewers.slice(0, 2).join(', ') +
      (viewers.length > 2 ? ` and ${viewers.length - 2} other` : '');

    logger.debug(`Set View: ${match.match_id}, ${viewers.length}, ${viewer_names}`);

    await matchModel.updateById(match.match_id, {
      viewer_cnt: viewers.length,
      viewer_names
    });

    res.result = [setView];
    return next();
  })
};
