
var Promise = require('bluebird');

var logger = require('../api/logger');
var config = require('../config.js');
var error = require('../classes/error.js');
var UserSocialModel = require('../models/user_social.js');
// var DeviceModel = require('../models/device.js');
const cache = require('bebo-node-commons').RedisCache;
var uuid = require('uuid');

var TwitchAPI = require('../classes/twitch_api_helix.js');
var TwitchAPIV5 = require('twitch-api');

var twitch = new TwitchAPI({
  clientId: config.TWITCH_CLIENT_ID,
  clientSecret: config.TWITCH_SECRET,
  redirectUri: config.TWITCH_REDIRECT_URI
});

var twitchV5 = new TwitchAPIV5({
  clientId: config.TWITCH_CLIENT_ID,
  clientSecret: config.TWITCH_SECRET
});



// 3min because otherwise we run into issues with people having to do captcha / 2fa
var EXPIRE = 60 * 3;

function setState(state, sData) {
  // logger.info('setState', state, sData);
  return cache
    .getWriteClient(config.CONNECT_REDIS_DB)
    .setAsync(state, sData)
    .then(function () {
      cache.getWriteClient(config.CONNECT_REDIS_DB).expireAsync(state, EXPIRE);
      return state;
    });
}

function getUserIdFromCode(code) {
  return cache
    .getReadClient(config.CONNECT_REDIS_DB)
    .getAsync(code)
    .then(data => {
      return JSON.parse(data);
    });
}


function saveTwitchUser(user_id, token_data, isStreamer) {

  let body = {};

  return twitchV5.getAuthenticatedUser(token_data.access_token)
    .then(_body => {
      body = _body;
      logger.debug('authenticated user', body);
      const twitch_id = body._id.toString();
      return UserSocialModel.getTwitchUserTuple(user_id, twitch_id);
    })
    .then(({user, us}) => {
      return twitchV5.getAuthenticatedUserChannel(token_data.access_token)
        .then(_body => {
          logger.debug("Authenticated Channel", _body);
          body = Object.assign(body, _body);
          logger.debug("merged", body);
          return {user,us};
        });
    })
    .then(({user, us}) => createOrUpdateTwitchUser(user, token_data, body, isStreamer, us))
    .then(us => {
      return us.user_id;
    });

}

function createOrUpdateTwitchUser(user, token_data, body = {}, isStreamer, us) {
  // logger.info("createOrUpdateTwitchUser", isStreamer);
  const scopes = twitch.getScopes(isStreamer);
  if (!us) {
    us = {
      user_id: user.user_id,
      social_id: body._id,
      data: body,
      scopes
    };
  } else {
    us.data = Object.assign(us.data, body);
  }

  logger.debug('token data', JSON.stringify(token_data));
  us.token = token_data.access_token;

  if (token_data.expires_in) {
    us.token_expires_dttm = new Date(Date.now() + token_data.expires_in);
  } else {
    us.token_expires_dttm = null;
  }

  us.refresh_token = token_data.refresh_token;

  us.scopes = scopes;

  // logger.debug('createOrUpdateTwitchUser', JSON.stringify(us));
  // return addImageUrlFromTwitch(us, isStreamer)
  return UserSocialModel.upsertWithDefaults(us);
}

var SocialTwitch = {
  route: '/social/twitch',

  get: function (req, res, next) {
    let query = req.query;

    //has to be here on top becasuse we need it in the error case,

    if (query.error) {
      logger.error(`error logging in`, query.error);
      res.redirect_url = `${config.WWW_URL}/oauth/twitch?error=${query.error}&message=${
        query.error_description
      }`;
      next();
      return;
    } else if (query.code && query.state) {
      // logger.info('state', query.state);
      // logger.info('query', query);

      let noKey = false;

      getUserIdFromCode(query.state)
        .then(data => {
          if (!data) {
            logger.error(
              `Failed to get user_id from code -- state: ${query.state} event_id: (${req &&
                req._event &&
                req._event.id})`
            );
            throw new error.BadRequest('failed_to_validate_code');
          }

          const user_id = data.user_id;
          noKey = data.no_key;
          const isStreamer = !data.no_key;

          return Promise.props({
            user_id: user_id,
            token: twitch.validateTwitchCode(query.code),
            noKey,
            isStreamer
          });
        })
        .then(props =>
          saveTwitchUser(props.user_id, props.token, props.isStreamer).then(user_id =>
            Object.assign({}, props, {
              user_id
            })
          )
        )
        .then(props => {
          const {
            noKey
          } = props;
          const correct_user_id = props.user_id;
          res.redirect_url = `${config.WWW_URL}/close`;
          logger.debug('redirecting to ', res.redirect_url);
          next();
        })
        .catch(err => {
          logger.error('failed to connect twitch', err);
          res.redirect_url = `${config.WWW_URL}/signup-error/${noKey ? 'viewer' : 'streamer'}/${
            err.message
          }`;
          next();
        });
    } else {
      if (!req.acting_user) {
        return next(new error.Forbidden("need a user, if you're local connect on dev first"));
      }

      let user_id = req.acting_user.user_id;

      // start twitch OIDC Authorization Code Flow (ID Tokens and User Access Tokens)
      let no_key = Boolean(req.query.no_key);

      var state = uuid.v4();
      var data = {
        user_id,
        no_key
      };
      const scopes = twitch.getScopes(!no_key);

      setState(state, JSON.stringify(data))
        .then(state => {
          res.redirect_url = twitch.getAuthorizationUrl({
            scopes,
            state,
            force_verify: 'true'
          });
          logger.debug('REDIRECT URL', res.redirect_url);
          next();
        })
        .catch(err => {
          logger.error('failed to validate token, save to redis, etc', err);
          next(err);
        });
    }
  }
};
module.exports = SocialTwitch;
