const { google } = require('googleapis');
const OAuth2 = google.auth.OAuth2;
const config = require('../config.js');
const logger = require('../api/logger');
const cache = require('bebo-node-commons').RedisCache;
const uuid = require('uuid');
const util = require('util');
const _ = require('lodash');
const UserSocialModel = require('../models/user_social.js');

var EXPIRE = 60 * 3;

function getState(code) {
  return cache
    .getReadClient(config.CONNECT_REDIS_DB)
    .getAsync(code)
    .then(data => {
      return JSON.parse(data);
    });
}

function setState(state, data) {
  logger.info('setState', state, data);
  return cache
    .getWriteClient(config.CONNECT_REDIS_DB)
    .setAsync(state, data)
    .then(function() {
      cache.getWriteClient(config.CONNECT_REDIS_DB).expireAsync(state, EXPIRE);
      return state;
    });
}

async function saveYouTubeUser(user_id, token_data, oauth2Client) {

  let new_user = true;

  var service = google.youtube('v3');


  let response = await service.liveBroadcasts.list({
    auth: oauth2Client,
    part: 'id,snippet,contentDetails,status',
    broadcastType: 'persistent',
    mine: true
  });

  let broadcast = null;
  if (response.data && response.data.items && response.data.items.length > 0) {
    broadcast = _.find(response.data.items, o => o.snippet.isDefaultBroadcast);
  }
  if (!broadcast) {
    logger.error('NO LIVE BROADCASTS ENABLED');
  }

  let boundStreamId = broadcast.contentDetails.boundStreamId;
  response = await service.liveStreams.list({
    auth: oauth2Client,
    part: 'id,snippet,cdn,status',
    id: boundStreamId
  });
  let stream = _.find(response.data.items, o => o.snippet.isDefaultStream);

  let data = {
    ingest: {
      type: stream.cdn.ingestionType,
      key: stream.cdn.ingestionInfo.streamName,
      address: stream.cdn.ingestionInfo.ingestionAddress,
      backup_address: stream.cdn.ingestionInfo.backupIngestionAddress
    },
    channel_id: stream.snippet.channelId,
    resolution: stream.cdn.resolution,
    frame_rate: stream.cdn.frameRate
  };

  var people = google.people('v1');
  response = await people.people.get({
    resourceName: 'people/me',
    personFields: 'emailAddresses,names,birthdays',
    auth: oauth2Client
  });
  let me = response.data;
  let social_id = me.resourceName.replace('people/', '');

  let name = _.find(me.names, o => o.metadata.primary);
  data.display_name = name.displayName;
  let email = _.find(me.emailAddresses, o => o.metadata.primary);
  data.email = email.value;

  return UserSocialModel.getYouTubeUserTuple(user_id, social_id)
    .then(({ user, us }) => {
      if (user.username) {
        new_user = false;
      }
      return {
        user,
        us
      };
    })
    .then(({ user, us }) => createOrUpdateYouTubeUser(user, token_data, social_id, data, true, us))
    .then(us => {
      return us.user_id;
    });
}

function createOrUpdateYouTubeUser(user, token_data, social_id, data = {}, isStreamer, us) {

  if (!us) {
    us = {
      user_id: user.user_id,
      stream_id: user.stream_id,
      social_id: social_id,
      type: 'youtube'
    };
  }
  us.data = data;
  us.token = token_data.access_token;

  if (token_data.expiry_date) {
    us.token_expires_dttm = new Date(token_data.expiry_date);
  } else {
    us.token_expires_dttm = null;
  }

  us.refresh_token = token_data.refresh_token;
  us.scopes = token_data.scope.split(' ');

  return UserSocialModel.upsertWithDefaults(us);
}

const SocialYouTube = {
  route: '/social/youtube',

  get: async function(req, res, next) {
    if (!req.acting_user) {
      return next(new error.Forbidden("need a user, if you're local connect on dev first"));
    }

    let user_id = req.acting_user.user_id;

    let query = req.query;

    //has to be here on top because we need it in the error case,
    let no_key = Boolean(req.query.no_key);

    if (query.error) {
      logger.error(`error logging in`, query.error);
      res.redirect_url = `${config.WWW_URL}/oauth/twitch?error=${query.error}&message=${
        query.error_description
      }`;
      next();
      return;
    } else if (query.code && query.state) {
      var oauth2Client = new OAuth2(
        config.YOUTUBE_CLIENT_ID,
        config.YOUTUBE_CLIENT_SECRET,
        config.YOUTUBE_REDIRECT_URIS[0]
      );
      const { tokens } = await oauth2Client.getToken(query.code);
      oauth2Client.credentials = tokens;

      getState(query.state)
        .then(data => {
          if (!data) {
            logger.error(
              `Failed to get user_id from code -- state: ${query.state} event_id: (${req &&
                req._event &&
                req._event.id})`
            );
            throw new error.BadRequest('failed_to_validate_code');
          }

          const uid = data.user_id;
          return saveYouTubeUser(data.user_id, tokens, oauth2Client).then(() => {
            res.redirect_url = `${config.WWW_URL}/oauth/youtube`;
            logger.debug('redirecting to ', res.redirect_url);
            next();
          });
        })
        .catch(err => {
          logger.error('failed to connect youtube', err);
          // wtf url encoding
          res.redirect_url = `${config.WWW_URL}/oauth/youtube?error_code=42&error=${err.message}`;
          next();
        });

    } else {
      // YouTube / Google OAuth2 flow

      var oauth2Client = new OAuth2(
        config.YOUTUBE_CLIENT_ID,
        config.YOUTUBE_CLIENT_SECRET,
        config.YOUTUBE_REDIRECT_URIS[0]
      );

      let state = uuid.v4();
      let data = {
        user_id
      };

      setState(state, JSON.stringify(data))
        .then(() => {
          var authUrl = oauth2Client.generateAuthUrl({
            access_type: 'offline',
            state: state,
            prompt: 'consent',
            defer: true,
            scope: config.YOUTUBE_STREAMER_SCOPES
          });
          res.redirect_url = authUrl;
          logger.debug('REDIRECT URL', res.redirect_url);
          next();
        })
        .catch(err => {
          logger.error('failed to save to redis', err);
          next(err);
        });
    }
  }
};
module.exports = SocialYouTube;
