const uuidv4 = require('uuid/v4');

const TarlyController = require('../classes/tarly_controller');
const InviteController = require('../controllers/invite');
const TeamController = require('../controllers/team');

const error = require('../classes/error.js');
const asyncMiddleware = require('../classes/async-middleware.js');
const UserController = require('../controllers/user');

const ALLOWED = new Set(
  ['team_id', 'name', 'image_url']);

module.exports = {
  route: '/team',
  post: asyncMiddleware(async function(req, res, next) {

    const LeagueModel = TarlyController.getModel('league');
    const UserLeagueModel = TarlyController.getModel('user_league');
    const userTeamModel = TarlyController.getModel('user_team');
    const teamModel = TarlyController.getModel('team');

    if (!req.acting_user) {
      throw new error.Unauthorized('Not Authenticated');
    }

    const user_id = req.acting_user.user_id;
    const { league_id, image_url = null, name = null } = req.body;

    if (!league_id) {
      return next(new error.BadRequest("league_id is required"));
    }

    let userLeague = await UserLeagueModel.model.findOne(
      { where: { user_id, league_id },
        paranoid: true,
        useMaster: false,
        raw: false,
        plain: true
      });
    if (!userLeague) {
      return next(new error.BadRequest('user needs to be in league'));
    }

    let existingTeam = await userTeamModel.model.findOne(
      { where: { league_id, user_id },
        paranoid: true,
        raw: false,
        plain: true
      });

    if (existingTeam) {
      return next(new error.Conflict('already in team for this league'));
    }

    const league = await LeagueModel.findByPk(league_id);

    let team = await TeamController.createAndJoinTeam({
      owner_id: user_id,
      team_id: uuidv4(),
      name: name,
      image_url : image_url,
      league_id: league_id,
      active_division_id: league.active_division_id
    }, user_id);

    let invite_url = await InviteController.getInviteUrl(user_id,
      "team_url",
      { team_id: team.team_id,
        league_id: league_id,
        media: "link"});

    await teamModel.model.update(
      {invite_url: invite_url},
      {where: { team_id: team.team_id }});

    team = await teamModel.model.findByPk(team.team_id, {
      include: teamModel.config.include,
      paranoid: true,
      useMaster: true,
      raw: false
    });

    UserController.emit(user_id);

    team = team.get();
    team.invite_url = invite_url;
    res.result = [team];
    next();
  }),
  put: asyncMiddleware(async function(req, res, next) {

    let team_id = req.body.team_id;
    if (! team_id) {
      return next(new error.BadRequest("team_id"));
    }
    let owner_id = undefined;
    if (!req.is_admin) {
      delete req.body.league_id;
      for (let key of Object.keys(req.body)) {
        if (! ALLOWED.has(key)) {
          return next(new error.Forbidden(`user forbidden to change ${key}`));
        }
      }
      owner_id = req.acting_user.user_id;
    } 

    const TeamModel = TarlyController.getModel('team');

    let updateObj = Object.assign({owner_id: owner_id}, req.body);

    let team = await TeamModel.updateById(team_id, updateObj, false);
    if (! team) {
      return (next(new error.NotFound("team not found")));
    }
    
    await TeamController.emitTeamUsers(team_id);
    res.result = [team];
    next();
  }),
  get: asyncMiddleware(async function(req, res, next) {
    let team_id = req.query.team_id;
    const teamModel = TarlyController.getModel('team');
    let team = await teamModel.model.findByPk(team_id, {
      include: teamModel.config.include,
      paranoid: true,
      useMaster: true,
      raw: false
    });
    team = team.get();
    res.result = [team];
    next();
  }),
  delete: asyncMiddleware(async function(req, res, next) {
    let team_id = req.query.team_id;

    if (! team_id) {
      return next(new error.BadRequest("need team_id"));
    }

    if (! req.is_admin) {
      return next(new error.Forbidden("Admin only at this point"));
    }
    const teamModel = TarlyController.getModel('team');
    await teamModel.model.destroy({where: {team_id}});
    const userTeamModel = TarlyController.getModel('user_team');
    await userTeamModel.model.destroy({where: {team_id}});
    let team = await teamModel.model.findByPk(team_id, {
      include: teamModel.config.include,
      paranoid: false,
      useMaster: true,
      raw: false
    });
    team = team.get();
    TeamController.emitTeamUsers(team_id);
    res.result = [team];
    next();
  }),
};

