const _ = require('lodash');
const isEmail = require('validator/lib/isEmail');
const error = require('../classes/error.js');
const TarlyController = require('../classes/tarly_controller');
const asyncMiddleware = require('../classes/async-middleware.js');
const SMS = require('../classes/sms');

const UserController = require('../controllers/user');
const FILTER_INVITE_KEYS = new Set(['created_dttm', 'updated_dttm', 'owner_id', 'url', 'code', 'expire_dttm', 'deleted_dttm']);

const INVITE_TO_CAMPAIGN = {
  "campaign_name": "name",
  "campaign_content": "content",
  "campaign_source": "source",
  "campaign_medium": "medium",
};

let UserModel;
let InviteModel;

function inviteToCampaign(invite) {
  let campaign = {};
  for (let [k, v] of Object.entries(invite)) {
    if (v == null || v === "") {
      continue;
    }
    if (FILTER_INVITE_KEYS.has(k)) {
      continue;
    } else if (INVITE_TO_CAMPAIGN[k]) {
      campaign[INVITE_TO_CAMPAIGN[k]] = v;
    } else {
      campaign[k] = v;
    }
  }
  return campaign;
}

const UTM_TO_CAMPAIGN = {
  "utm_campaign": "name",
  "utm_medium": "medium",
  "utm_content": "content",
  "utm_source": "source", 
  "utm_term": "term"
};

function utmToCampaign(body) {
  let campaign = {};

  for (let [utmArg, campaignKey] of Object.entries(UTM_TO_CAMPAIGN)) {
    let value = body[utmArg];
    if (value == null) {
      continue;
    }
    value = _.trim(value);
    if (value === "") {
      return;
    }
    campaign[campaignKey] = value;
  }
  return campaign;
}

async function getCampaign(body) {
  let campaign = {name: "organic"};
  if (body.code) {
    let invite = await InviteModel.findOne({ code:body.code }, false, false);
    if (invite) {
      campaign = inviteToCampaign(invite.get());
    }
  }
  let utmCampaign = utmToCampaign(body);
  campaign = Object.assign(campaign, utmCampaign);
  return campaign;

}


module.exports = {
  route: '/user/signup',
  post: asyncMiddleware(async function(req, res, next) {
    UserModel = UserModel || TarlyController.getModel('user');
    InviteModel = InviteModel || TarlyController.getModel('invite');

    if (!req.body) {
      return next(new error.BadRequest('no body'));
    }

    let { username, password, email, phone_number, pin } = req.body;

    const campaign = await getCampaign(req.body);
    const phone = SMS.parsePhoneNumber(phone_number);

    if (phone_number) {
      if (!phone) {
        return next(new error.BadRequest('invalid phone number'));
      }
    }

    if (email && !isEmail(email)) {
      email = null;
    }

    if (!phone && !email) {
      return next(new error.BadRequest('missing valid phone or email'));
    }

    if (phone && !username) {
      if (! pin) {
        //start of sms flow
        await UserController.sendValidationSMS(phone, campaign);
        let message = UserController.formatPinMessage(phone);
        res.result = [{ message, require: ["pin"]}];
        res.code = 202;
        return next();

      } else {

        let user = null;
        try {
          user = await UserController.loginWithPhoneNumber(phone, pin, req.platform);
        } catch (err) {
          user = await UserController.signupWithPhoneNumber(phone, pin, campaign, req.platform);
        }
        res.result = [user];
        res.acting_user = user;
        return next();
      }
    }

    if (!password || !username) {
      return next(new error.BadRequest('phone_number or username/password required'));
    }

    const user = await UserController.signupWithPassword(username, password, phone, email, campaign, req.platform);
    res.result = [user];
    res.acting_user = user;
    return next();

  })
};
