const cache = require('bebo-node-commons').RedisCache;
const error = require('../classes/error.js');
const asyncMiddleware = require('../classes/async-middleware.js');
const AuthenticatedUser = require('../middleware/authenticated_user');
const _ = require('lodash');

const config = require('../config');
const writer = cache.getWriteClient(config.USER_STATE_REDIS_DB);
const reader = cache.getReadClient(config.USER_STATE_REDIS_DB);

function getKey(user, service) {
  return `user_state:${service}_${user}`;
}

const UserState = {
  route: '/user/state/:service',
  wrapper: AuthenticatedUser,
  put: asyncMiddleware(async function(req, res, next) {
    const { user_id } = req.acting_user;
    const { service } = req.params;
    const expire = parseInt(req.body.expire);

    if (!expire || isNaN(expire)) {
      return next(new error.BadRequest('Request must have a valid expire time'));
    }

    const data = req.body.data || {};

    data.user_id = user_id;
    data.exist = true;

    const raw = await writer.getAsync(getKey(user_id, service));
    const isSame = _.isEqual(data, JSON.parse(raw));
    if(!isSame){
      res.result = [data];
    }

    await writer.setAsync(getKey(user_id, service), JSON.stringify(data), 'EX', expire);
    next();
  }),

  get: asyncMiddleware(async function(req, res, next) {
    if (!req.acting_user) {
      return next(new error.Forbidden('No user'));
    }

    const { user_id } = req.acting_user;
    const { service } = req.params;

    const raw = await reader.getAsync(getKey(user_id, service));

    const data = JSON.parse(raw) || {
      exist: false,
      user_id
    };

    res.result = [data];
    next();
  }),

  delete: asyncMiddleware(async function(req, res, next) {
    if (!req.acting_user) {
      return next(new error.Forbidden('No user'));
    }

    const { user_id } = req.acting_user;
    const { service } = req.params;

    await writer.delAsync(getKey(user_id, service));

    const data = {
      exist: false,
      user_id
    };

    res.result = [data];
    next();
  })
};

UserState.post = UserState.put;

module.exports = UserState;
