const crypto = require('crypto');
const config = require('../config');
const Cache = require('bebo-node-commons').RedisCache;
const logger = require('../api/logger');
const cacheRead = Cache.getReadClient(config.GLOBAL_REQUEST_CACHE);
const cacheWrite = Cache.getWriteClient(config.GLOBAL_REQUEST_CACHE);

const getFromCache = (req, res, next) => {
  if (!req.params) {
    req.params = {};
  }
  if (!req.query) {
    req.query = {};
  }
  // otherwise we're going to check the global cache
  const cache_key = crypto
    .createHash('md5')
    .update(`${req.path}_${JSON.stringify(req.query)}_${JSON.stringify(req.params)}`)
    .digest('hex');

  req.cache_key = cache_key;

  return cacheRead
    .getAsync(cache_key)
    .then(result => {
      if (result) {
        try {
          res.from_cache = true;
          res.merged = { cache_key };
          res.result = JSON.parse(result);
        } catch (err) {
          return Promise.reject(err);
        }
      }
      return next();
    })
    .catch(err => {
      logger.error(
        'failed to get data from cache, continuing on to fetch data from DB',
        err,
        err && err.stack
      );
      return next();
    });
};

const saveToCache = seconds => {
  return (req, res, next) => {
    if (req.cache_key && !res.from_cache) {
      return cacheWrite.setAsync(req.cache_key, JSON.stringify(res.result)).then(() => {
        cacheWrite.expireAsync(req.cache_key, seconds);
        return next();
      });
    }
    return next();
  };
};

const nextWrapper = nextFn => {
  return (req, res, next) => {
    if (res.result) {
      return next();
    }
    return nextFn(req, res, next);
  };
};

//exports a factory that...
module.exports = cacheSeconds => {
  //...wraps the main function with 2 pieces of middleware that handle caching for us generically
  return { getFromCache, nextWrapper, saveToCache: saveToCache(cacheSeconds) };
};
