const Promise = require('bluebird');
const Sequelize = require('sequelize');
const uuid = require('uuid');

const sqlPool = require('../classes/sql_pool.js');
const config = require('../config.js');

const GET_DEPLOYED_DETECTORS_QUERY = `SELECT bran_detector.*, bran_deploy.* 
FROM bran_detector
INNER JOIN bran_deploy
ON bran_detector.id=bran_deploy.detector_id AND bran_deploy.bebo_env=:bebo_env;`;

const GET_LATEST_DETECTOR = `
SELECT * FROM bran_detector where game=:game AND type=:type ORDER BY string_to_array(version, '.')::int[] DESC LIMIT 1;
`;

const bran_detector_definition = {
  id: {
    type: Sequelize.TEXT,
    primaryKey: true
  },
  bebo_env: Sequelize.TEXT,
  type: Sequelize.TEXT,
  game: Sequelize.TEXT,
  version: Sequelize.TEXT,
  detector_file: Sequelize.TEXT,
  created_dttm: Sequelize.DATE,
  updated_dttm: Sequelize.DATE
};

const BranDetector = sqlPool.define('bran_detector', bran_detector_definition, {
  timestamps: false,
  tableName: 'bran_detector'
});

const bran_deploy_definition = {
  type: {
    type: Sequelize.TEXT,
    primaryKey: true
  },
  game: {
    type: Sequelize.TEXT,
    primaryKey: true
  },
  bebo_env: {
    type: Sequelize.TEXT,
    primaryKey: true
  },
  detector_id: Sequelize.TEXT,
  created_dttm: Sequelize.DATE,
  updated_dttm: Sequelize.DATE
};

const BranDeploy = sqlPool.define('bran_deploy', bran_deploy_definition, {
  timestamps: false,
  tableName: 'bran_deploy'
});

const getEnv = () => {
  if (config.isLocal() || config.env === "greengoblin") {
    return 'dev';
  }
  return config.env;
};

BranDetector.getLatestVersion = (game, type) => {
  const replacements = { game, type };
  return sqlPool.query(GET_LATEST_DETECTOR, {type: Sequelize.QueryTypes.SELECT, replacements}).then(detectors => {
    if(detectors.length) {
      return detectors[0].version;
    }

    return "0.0.0";
  });
};

BranDetector.getNextPatchVersion = (game, type) => {
  return BranDetector.getLatestVersion(game, type).then(version => {
    const version_split = version.split('.').map(s => parseInt(s));
    const major = version_split[0];
    const minor = version_split[1];
    const patch = version_split[2] + 1;
    return `${major}.${minor}.${patch}`;
  });
};

BranDetector.getNextMinorVersion = (game, type) => {
  return BranDetector.getLatestVersion(game, type).then(version => {
    const version_split = version.split('.').map(s => parseInt(s));
    const major = version_split[0];
    const minor = version_split[1] + 1;
    const patch = 0;
    return `${major}.${minor}.${patch}`;
  });
};

BranDetector.getNextMajorVersion = (game, type) => {
  return BranDetector.getLatestVersion(game, type).then(version => {
    const version_split = version.split('.').map(s => parseInt(s));
    const major = version_split[0] + 1;
    const minor = 0;
    const patch = 0;
    return `${major}.${minor}.${patch}`;
  });
};

BranDetector.get = () => {
  const replacements = { bebo_env: getEnv() };
  return sqlPool.query(GET_DEPLOYED_DETECTORS_QUERY, {replacements}).then(detectors => {
    return detectors[0];
  });
};

BranDetector.post = new_detector => {
  const set_to_live = new_detector.live;
  const {game, type} = new_detector;
  new_detector.id = uuid.v4();
  new_detector.bebo_env = getEnv();

  const bump = new_detector.bump;

  if(!bump) {
    throw new Error("must pass in bump: {major|minor|patch}");
  }

  let next_version_prom = null;

  if(bump === "major") {
    next_version_prom = BranDetector.getNextMajorVersion(game, type);
  } else if (bump === "minor") {
    next_version_prom = BranDetector.getNextMinorVersion(game, type);
  } else if (bump === "patch") {
    next_version_prom = BranDetector.getNextPatchVersion(game, type);
  } else {
    throw new Error("must pass in bump: {major|minor|patch}");
  }

  return next_version_prom.then(version => {
    new_detector.version = version;
    delete new_detector.bump;
    return BranDetector.create(new_detector).then(() => {
      if(set_to_live) {
        const deploy = { 
          bebo_env: getEnv(),
          game: new_detector.game,
          detector_id: new_detector.id,
          type: new_detector.type
        };
        return BranDeploy.upsert(deploy).then(() => new_detector);
      }
      return Promise.resolve(new_detector);
    });
  });

};

module.exports = BranDetector;
