const Promise = require('bluebird');

const TarlyController = require('../classes/tarly_controller');
const TarlySQLPool = require('../classes/sql_pool');
const logger = require('../api/logger');
const cache = require('bebo-node-commons').RedisCache;
const config = require('../config.js');

const EXPIRE = 60 * 60 * 24 * 5;
const CACHE_KEY = 'ALL_SERVICES_' + config.env;

const GET_SERVICES_QUERY = `SELECT service.*
   FROM service
   INNER JOIN
   live_service ON service.name=live_service.name AND service.version=live_service.version
   WHERE
   service.bebo_env=:bebo_env`;

const get_services_from_cache = () => {
  logger.debug('Getting services from cache');
  return cache
    .getReadClient(config.GLOBAL_REQUEST_CACHE)
    .getAsync(CACHE_KEY)
    .then(services => {
      if (!services) {
        logger.info('Services not cached');
        return Promise.resolve();
      }
      logger.debug('Found services in cache');
      return Promise.resolve(JSON.parse(services));
    })
    .catch(() => {
      logger.error('Error getting services from cache');
      return Promise.resolve();
    });
};

const set_services_to_cache = services => {
  logger.info('Setting services to cache');
  const writeClient = cache.getWriteClient(config.GLOBAL_REQUEST_CACHE);
  return writeClient
    .setAsync(CACHE_KEY, JSON.stringify(services))
    .then(() => {
      return writeClient.expireAsync(CACHE_KEY, EXPIRE);
    })
    .then(() => {
      logger.info('Successfully set services to cache');
      return Promise.resolve();
    })
    .catch(err => {
      logger.info('Error setting services to cache: ', err);
      return Promise.reject('Error setting services to cache');
    });
};

const bust_cache = () => {
  logger.info('Busting services cache');
  const writeClient = cache.getWriteClient(config.GLOBAL_REQUEST_CACHE);
  return writeClient.delAsync(CACHE_KEY);
};
module.exports = {
  get: () => {
    return get_services_from_cache().then(services => {
      if (!services) {
        return TarlySQLPool.query(GET_SERVICES_QUERY, {
          replacements: { bebo_env: config.env }
        }).then(([db_services]) => {
          return set_services_to_cache(db_services)
            .then(() => db_services)
            .catch(err => {
              return Promise.resolve(db_services);
            });
        });
      }
      logger.debug('returning cached services');
      return services;
    });
  },

  post: new_service => {
    const Service = TarlyController.getModel('service');
    const LiveService = TarlyController.getModel('live_service');

    const bebo_env = config.env;
    const where_name_bebo_env = { name: new_service.name, bebo_env };
    const set_to_live = new_service.live || false;
    new_service.bebo_env = bebo_env;
    return Service.findOne(where_name_bebo_env)
      .then(existing => {
        if (existing) {
          return Service.updateWhere(new_service, where_name_bebo_env);
        }
        return Service.create(new_service);
      })
      .then((service) => {
        if (set_to_live) {
          return LiveService.findOne(where_name_bebo_env)
            .then(live_service => {
              if (live_service) {
                return LiveService.updateWhere({ version: new_service.version }, where_name_bebo_env);
              }

              return LiveService.create({
                owner_id: new_service.owner_id,
                name: new_service.name,
                version: new_service.version,
                bebo_env
              });
            }).then(() => {
              return service;
            });
        }

        return service;
      })
      .then((service) => {
        return bust_cache()
          .then(() => service);
      });
  }
};
