const Sequelize = require('sequelize');
const { Op } = Sequelize;
const sqlPool = require('../classes/sql_pool.js');
const config = require('../config.js');
const logger = require('../api/logger');
const Promise = require('bluebird');
const User = require('../api/user');
const error = require('../classes/error.js');

const TwitchAPI = require('../classes/twitch_api_helix.js');
const twitch = new TwitchAPI({
  clientId: config.TWITCH_CLIENT_ID,
  clientSecret: config.TWITCH_SECRET,
  redirectUri: config.TWITCH_REDIRECT_URI,
  scopes: config.TWITCH_STREAMER_SCOPES
});


var UserSocial = sqlPool.define(
  'user_social',
  {
    user_id: {
      type: Sequelize.TEXT,
      primaryKey: true
    },
    token: Sequelize.TEXT, // access_token
    refresh_token: Sequelize.TEXT,
    token_expires_dttm: Sequelize.DATE,
    token_secret: Sequelize.TEXT,
    type: {
      type: Sequelize.TEXT,
      primaryKey: true
    },
    resolution: Sequelize.TEXT,
    bitrate: Sequelize.INTEGER,
    twitch_bitrate: Sequelize.INTEGER,
    fps: Sequelize.INTEGER,
    data: Sequelize.JSONB,
    social_id: Sequelize.TEXT,
    stream_id: Sequelize.TEXT,
    active: Sequelize.BOOLEAN,
    live: Sequelize.BOOLEAN,
    bot_enable: Sequelize.BOOLEAN,
    scopes: Sequelize.ARRAY(Sequelize.TEXT),
    transport: Sequelize.TEXT,
    codec: Sequelize.TEXT,
    codec_settings: Sequelize.JSONB,
    app_admin_mode: Sequelize.BOOLEAN,
    encoder: Sequelize.TEXT,
    encoder_preset: Sequelize.TEXT,
    updated_dttm: Sequelize.DATE,
    created_dttm: Sequelize.DATE,
    max_upload: Sequelize.INTEGER,
    migration_dttm: Sequelize.DATE
  },
  {
    timestamps: false,
    tableName: 'user_social'
  }
);

const addTwitchContext = (data, us) => {
  if (us.data) {
    data.followers = us.data.followers;
    data.channel_name = us.data.name;
    data.live = us.live;
    data.bot_enable = us.bot_enable;
    data.cover_image_url = us.data.profile_banner;
  } else {
    logger.warn("us.data isn't there, that's a problem");
  }

  return data;
};

const addTwitterData = (data, us) => {
  data.followers = us.data.followers;
  data.screen_name = us.data.screen_name;

  return data;
};

const publicUserSocialObject = (us) => {
  if (!us) return Promise.reject(null);

  var data = {
    type: us.type,
    social_id: us.social_id,
    stream_id: us.stream_id,
    fps: us.fps,
    resolution: us.resolution,
    bitrate: us.bitrate === 0 ? 'default' : us.bitrate,
    twitch_bitrate: us.twitch_bitrate === 0 ? 'default' : us.twitch_bitrate,
    scopes: us.scopes,
    live: us.live,
    app_admin_mode: us.app_admin_mode,
    max_upload: us.max_upload
  };

  if (data.type === 'twitch') {
    data = addTwitchContext(data, us);
  } else if (data.type === 'twitter') {
    data = addTwitterData(data, us);
  }
  return data;
};

UserSocial.addTwitchContext = addTwitchContext;

UserSocial.setScopes = (user_id, type, scopes) => {
  if (!user_id || typeof user_id !== 'string') {
    return Promise.reject('user_id must be string');
  }
  if (!type || typeof type !== 'string') {
    return Promise.reject('type must be string');
  }
  if (!scopes || !Array.isArray(scopes)) {
    return Promise.reject('scopes must be an array');
  }

  return sqlPool.transaction(transaction =>
    UserSocial.update(
      {
        scopes
      },
      {
        where: {
          user_id,
          type
        },
        transaction
      }
    ).then(() => UserSocial.getSocialRaw(user_id, type, transaction))
  );
};

UserSocial.convertToPublicObject = (us, device_id = null, stream_id = null) => {
  return publicUserSocialObject(us, device_id).then(us => {
    us.stream_id = stream_id;
    return us;
  });
};

UserSocial.getResolutionByStreamId = (stream_id, transaction) => {
  return UserSocial.findOne({
    where: {
      stream_id,
      type: 'twitch'
    },
    transaction
  }).then(broadcast => {
    return broadcast.resolution;
  });
};

UserSocial.getBroadcastByStreamId = (stream_id, device_id = null, transaction) => {
  if (!stream_id) {
    return Promise.reject(new Error('no stream id in getBroadcastByStreamId'));
  }
  return UserSocial.findOne({
    where: {
      stream_id,
      type: 'twitch'
    },
    transaction
  })
    .then(us => UserSocial.convertToPublicObject(us, device_id, stream_id))
    .then(us => {
      if (us.codec === 'default') {
        delete us.codec;
      }
      if (us.transport === 'default') {
        delete us.transport;
      }
      return us;
    });
};

UserSocial.getSocial = (user_id, optional_type) => {
  let query = {
    user_id
  };
  if (optional_type) query.type = optional_type;
  return UserSocial.findAll({
    where: query
  }).then(user_socials => {
    return Promise.map(user_socials, us => publicUserSocialObject(us));
  });
};

UserSocial.getSocialRaw = (user_id, optional_type, transaction) => {
  let query = {
    user_id
  };
  if (optional_type) query.type = optional_type;
  return UserSocial.findOne({
    where: query,
    transaction
  });
};

UserSocial.getBySocialId = (social_id, optional_type, full) => {
  let query = {
    social_id
  };
  if (optional_type) query.type = optional_type;
  return UserSocial.findAll({
    where: query
  }).then(user_socials => {
    if (!full) {
      return Promise.map(user_socials, us => publicUserSocialObject(us));
    } else {
      return user_socials;
    }
  });
};

UserSocial.getBySocialIdRaw = (social_id, optional_type) => {
  let query = {
    social_id
  };
  if (optional_type) query.type = optional_type;
  //changed findAll to findOne... please do not break
  return UserSocial.findOne({
    where: query
  });
};
UserSocial.getByUserIdRaw = (user_id, optional_type) => {
  let query = {
    user_id
  };
  if (optional_type) query.type = optional_type;
  //changed findAll to findOne... please do not break
  return UserSocial.findOne({
    where: query
  });
};

UserSocial.changeLive = (arr = [], live = false) => {
  if (arr.length === 0) return Promise.reject('No arr passed to UserSocial.changeLive');
  let query = '';
  arr.forEach(stream_id => {
    query += `UPDATE user_social SET live = ${live} WHERE stream_id='${stream_id}' AND type='twitch'; `;
  });
  logger.debug(`Changing ${arr} live to ${live}`);
  return sqlPool.query(query);
};

UserSocial.findAllWhoLive = () => {
  return UserSocial.findAll({
    where: {
      type: 'twitch',
      token: {
        not: null
      },
      [Op.or]: [
        {
          live: true,
          stream_id: {
            not: null
          }
        }
      ]
    }
  });
};

UserSocial.getBroadcasterType = user_id => {
  return sqlPool
    .query(
      `SELECT data->>'broadcaster_type' as broadcaster_type
     FROM user_social
     WHERE type='twitch'
         AND user_id=:user_id
         AND deleted_dttm IS NULL;`,
      {
        type: Sequelize.QueryTypes.SELECT,
        replacements: { user_id }
      }
    )
    .then(rows => {
      if (!rows) {
        return '';
      }
      let result = rows.reduce((accumulator, currentValue) => {
        currentValue = currentValue.broadcaster_type;
        if (currentValue === 'partner' || accumulator === 'partner') {
          return 'partner';
        }
        if (currentValue === 'affiliate' || accumulator === 'affiliate') {
          return 'affiliate';
        }
        return '';
      }, '');
      return result;
    });
};

UserSocial.findPartnerOrAffiliateWhoLive = () =>
  sqlPool
    .query(
      `SELECT * FROM user_social WHERE type='twitch' AND data->'broadcaster_type' IS NOT NULL AND live = true AND data->>'broadcaster_type' != '';`
    )
    .then(([data]) => data);

UserSocial.setMediaConstraintsPublic = (stream, updateObj = {}, device_id) =>
  sqlPool.transaction(transaction =>
    UserSocial.setMediaConstraints(stream.creator.user_id, updateObj, transaction).then(() =>
      UserSocial.getBroadcastByStreamId(stream.stream_id, device_id, transaction)
    )
  );

UserSocial.setMediaConstraints = (user_id, updateObj = {}, transaction) => {
  let prom = () => Promise.resolve(updateObj);

  if ('bot_enable' in updateObj) {
    prom = () =>
      UserSocial.getSocialRaw(user_id, 'twitch', transaction).then(oldUs => {
        return updateObj;
      });
  }

  return prom()
    .then(theUpdate =>
      UserSocial.update(theUpdate, {
        where: {
          user_id,
          type: 'twitch'
        },
        transaction
      })
    )
    .then(() => {
      logger.debug('Constraints updated successfully for user', user_id);
    })
    .catch(err => {
      logger.error('Constraints update failed for user', user_id, err);
    });
};

UserSocial.getTwitchUserTuple = (user_id, twitch_id) => {
  return UserSocial.findOne({
    where: {
      social_id: twitch_id,
      user_id: user_id,
      type: 'twitch'
    }
  }).then(us => {
    if (us) {
      user_id = us.user_id;
    }
    return Promise.props({
      user: User.get(user_id),
      us
    });
  });
};

UserSocial.getYouTubeUserTuple = (user_id, social_id) => {
  // only works with existing users at the moment
  return User.get(user_id).then(user => {
    let where = {
      social_id: social_id,
      type: 'youtube'
    };
    if (user.username) {
      where.user_id = user_id;
    }
    return Promise.props({
      user: user,
      us: UserSocial.findOne({ where })
    });
  });
};

UserSocial.upsertWithDefaults = us => {
  if (us.dataValues) {
    us = us.dataValues;
  }

  //TODO @fpn @nick we should not hard code this to twitch, it should be explicitly passed in and if "type" does not exist in the data that is passed in we need to throw an error here
  const defaults = {
    type: 'twitch',
    active: true,
    bot_enable: true,
    updated_dttm: Sequelize.fn('NOW')
  };

  const update = Object.assign({}, defaults, us);
  logger.debug("upserting", update);
  return UserSocial.upsert(update).then(() =>
    UserSocial.getByUserIdRaw(update.user_id, update.type)
  );
};

function updateWithTwitchUserData(user_id, username, user_social, transaction) {
  let props = {};

  let update = {};

  if (username && user_social && user_social.data && user_social.data.display_name !== username) {
    const opts = {
      username: username
    };
    props.user = User.update(user_id, opts);
    update = {
      data: Object.assign({}, user_social.data, {
        display_name: username
      })
    };
  }
  if (user_social && user_social.refresh_token) {
    update.token = user_social.token;
    update.token_expires_dttm = user_social.token_expires_dttm;
    update.refresh_token = user_social.refresh_token;
  }
  if (update.data || update.token) {
    props.us = UserSocial.update(update, {
      where: {
        user_id,
        type: 'twitch'
      },
      transaction
    });
  }
  return Promise.props(props);
}

function checkValidationScopeMatch(validation, isStreamer) {
  if (!isStreamer) {
    return Promise.resolve();
  }
  if (!validation.scopes || !validation.scopes.includes('channel_editor')) {
    logger.info('user scope got downgraded, forcing re-auth');
    return Promise.reject(new error.Unauthorized('token_revoked'));
  }
  return Promise.resolve();
}

UserSocial.reAuthTwitch = (user_id, isStreamer) => {
  // FIXME: we should not create a database transaction and execute remote
  // calls while the transaction is open, this needs to handled programatically
  // and not on the db layer, so we need to remove this transaction:

  return sqlPool.transaction(transaction => {
    return UserSocial.getSocialRaw(user_id, 'twitch', transaction)
      .then(us => {
        // so we can test being expired
        // return twitch.revokeAccessToken(us.token, {user_id})
        //   .then(() => us);
        // })
        // .then(us => {

        if (us.refresh_token) {
          return twitch
            .refreshToken(us.refresh_token, {
              user_id
            })
            .then(token_data => {
              us.token = token_data.access_token;
              if (token_data.expires_in) {
                us.token_expires_dttm = new Date(Date.now() + 1000 * token_data.expires_in);
              } else {
                us.token_expires_dttm = null;
              }

              us.refresh_token = token_data.refresh_token;
              return us;
            });
        }
        return Promise.resolve(us);
      })
      .then(us =>
        Promise.props({
          us,
          validation: twitch.validateAccessToken(us.token, {
            user_id
          })
        })
      )
      .then(({ us, validation }) => {
        return checkValidationScopeMatch(validation, isStreamer)
          .then(() => updateWithTwitchUserData(user_id, validation.login, us, transaction))
          .then(() => UserSocial.getBroadcastByStreamId(us.stream_id, null, transaction));
      });
  });
};

module.exports = UserSocial;
