const process = require('process');
const config = require('./config.js');
const dispatcher = require('./api/dispatcher');
const logger = require('./api/logger');

async function App() {


  if (! config.initialized) {
    await config.init({name: "pharah"});
  }
  await logger.init(config);

  const Sentry = require('@sentry/node');
  if (config.env === 'bebo-prod' || config.env === 'dev') {
    Sentry.init({
      dsn: '**REMOVED**',
      environment: config.env,
      release: process.env.BEBO_VERSION
    });
  }

  process.on('uncaughtException', function(err) {
    // https://nodejs.org/api/process.html#process_warning_using_uncaughtexception_correctly
    // DON'T CALL logger here - syslog is async - only synchronous calls allowed here!
    console.error(new Date().toUTCString() + ' uncaughtException:', err.message);
    console.error(err.stack);
    process.exit(1);
  });

  process.on('unhandledRejection', (reason, p) => {
    logger.error('Unhandled Promise.reject at: Promise', p, 'reason:', reason);
  });

  process.on('warning', warning => {
    logger.warn('node.js:', warning.name, warning.message, warning.stack);
  });

  await dispatcher.init(config);

  const broadcast = require('./api/broadcast');
  await broadcast.init(config);

  const bodyParser = require('body-parser');
  const busboy = require('connect-busboy');
  const cookieParser = require('cookie-parser');
  const semver = require('semver');
  const express = require('express');
  const uuid = require('uuid');
  const auth = require('./classes/auth.js');
  const timeout = require('connect-timeout');
  const cors = require('cors');
  const http = require('http');
  const analytics = require('./classes/analytics.js');
  // const xhub = require('express-x-hub');
  const url = require('url');
  const TarlyController = require('./classes/tarly_controller');

  const Bot = require('./classes/bot-middleware.js');
  const generateCacheMiddleware = require('./middleware/cache');
  const queryToSQLOptionsMiddleware = require('./middleware/query_to_sql_options');
  const emitEvent = require('./middleware/emit_event');
  const User = require('./api/user');

  console.log("done requires");

  analytics.init();

  var app = express();


  function setIp(req, res, next) {
    if (config.env !== 'bebo-prod' && config.env !== 'dev' && !req.headers['x-real-ip']) {
      req.headers['x-real-ip'] = '0.0.0.0';
    }
    if (config.env !== 'bebo-prod' && config.env !== 'dev' && !req.headers['x-forwarded-for']) {
      req.headers['x-forwarded-for'] = '0.0.0.0';
    }
    next();
  }

  function setupEvent(req, res, next) {
    var now = new Date();
    req._event = {
      id: uuid.v4(),
      path: req.path,
      params: req.params,
      query: req.query,
      url: req.url
    };
    var ua = req.get('User-Agent');
    var platform;
    if (/iPhone/i.test(ua)) {
      platform = 'ios';
    } else if (/okhttp/i.test(ua)) {
      platform = 'android';
    } else if (/Android/i.test(ua)) {
      platform = 'android';
    } else if (/^Mozilla/i.test(ua)) {
      platform = 'web';
    } else if (/^Bebo/i.test(ua)) {
      platform = 'pc';
    }
    if (ua == null) {
      platform = 'pc';
    }
    req.platform = platform;

    if (ua) {
      var v = ua.match(/^[A-Za-z0-9_-]+\/([0-9.]+) /);
      if (v && v.length > 1) {
        var version = v[1];
        if (semver.valid(version)) {
          req.doBackwardsBefore = function(expectedPlatform, newVersion) {
            if (expectedPlatform !== platform) {
              return false;
            }
            if (semver.lt(version, newVersion)) {
              return true;
            }
            return false;
          };
        }
      }
    }
    if (req.doBackwardsBefore === undefined) {
      req.doBackwardsBefore = function() {
        return false;
      };
    }

    req._audit_event = {
      _id: req._event.id,
      event_id: req._event.id,
      platform_tx: platform,
      event_at: now.getTime(),
      event_dttm: now.toISOString(),
      start_at: now.getTime(),
      start_dttm: now.toISOString()
    };

    let u = req.originalUrl;
    u = url.parse(u);
    let path = u.pathname;
    // logger.debug("path", path);
    path = path.replace(/\/*$/g, '');
    let split = path.split('/');
    //TODO Fix this hack
    if (!app.isValidSubRoute(split[split.length - 1]) && !path.startsWith('/user/state')) {
      // so we will get false positives on 404
      split.splice(-1, 1);
      path = split.join('/');
    }

    req.originalPath = path;
    next();
  }

  function setupActiveHandler() {
    app.get('/active', function(req, res, next) {
      next();
    });
  }

  function setupHandlers() {
    // then we setup the handlers
    return new Promise((resolve, reject) => {
      var normalizedPath = require('path').join(__dirname, config.HANDLER_DIR);

      var paths = require('fs').readdirSync(normalizedPath);
      for (var p = 0; p < paths.length; p++) {
        var file = paths[p];
        if (!file.endsWith('.js')) {
          continue;
        }
        // logger.log('setting up handlers/' + file); // removing useless logs
        try {
          var handler = require(config.HANDLER_DIR + '/' + file);
        } catch (err) {
          logger.error('failed to require handler ' + file, err, err.stack);
          // FIXME:
          reject(err);
        }
        var route = handler.route;
        // logger.info('registering: ' + route); // removing useless logs
        // var routing_key = route.replace( /\//g, ".");
        app.reserveRoute(route);
        if (handler.get) {
          // logger.info(`routing_key GET${routing_key}`);
          let wrappers = [];
          let handlerFunction = null;
          //apply global wrapper if it exists
          if (handler.wrapper) {
            wrappers.push(handler.wrapper);
          }
          //if it's just a function then just register the function
          if (typeof handler.get === 'function') {
            handlerFunction = handler.get;
          } else if (handler.get.callback) {
            //we now expect an object with keys of callback AND / OR wrap
            //callback: fn
            //wrap?: [wrapper, ...]
            handlerFunction = handler.get.callback;
            wrappers = wrappers.concat(handler.get.wrap || []);
          }
          //do we need to do caching?
          let cacheGet = [];
          let cacheSave = [];
          if (handler.cache) {
            const disCache = generateCacheMiddleware(handler.cache);
            cacheGet = [disCache.getFromCache];
            handlerFunction = disCache.nextWrapper(handlerFunction);
            cacheSave = [disCache.saveToCache];
          }
          //finally just apply all the above
          //it does not matter if wrappers is an empty array, it will then just be ignored by express
          app.get(route, wrappers, cacheGet, handlerFunction, cacheSave);
        }

        if (handler.options) {
          // logger.info(`routing_key OPTIONS${routing_key}`);
          let wrappers = [];
          let handlerFunction = null;
          //apply global wrapper if it exists
          if (handler.wrapper) {
            wrappers.push(handler.wrapper);
          }
          //if it's just a function then just register the function
          if (typeof handler.options === 'function') {
            handlerFunction = handler.options;
          } else if (handler.options.callback) {
            //we now expect an object with keys of callback AND / OR wrap
            //callback: fn
            //wrap?: [wrapper, ...]
            handlerFunction = handler.options.callback;
            wrappers = wrappers.concat(handler.options.wrap || []);
          }
          //finally just apply all the above
          //it does not matter if wrappers is an empty array, it will then just be ignored by express
          app.options(route, wrappers, handlerFunction);
        } else {
          app.options(route);
        }

        if (handler.post) {
          // logger.info(`routing_key POST${routing_key}`);
          let wrappers = [];
          let handlerFunction = null;
          //apply global wrapper if it exists
          if (handler.wrapper) {
            wrappers.push(handler.wrapper);
          }
          //if it's just a function then just register the function
          if (typeof handler.post === 'function') {
            handlerFunction = handler.post;
          } else if (handler.post.callback) {
            //we now expect an object with keys of callback AND / OR wrap
            //callback: fn
            //wrap?: [wrapper, ...]
            handlerFunction = handler.post.callback;
            wrappers = wrappers.concat(handler.post.wrap || []);
          }
          //finally just apply all the above
          //it does not matter if wrappers is an empty array, it will then just be ignored by express
          app.post(route, wrappers, handlerFunction);
        }

        if (handler.put) {
          // logger.info(`routing_key PUT${routing_key}`);
          let wrappers = [];
          let handlerFunction = null;
          //apply global wrapper if it exists
          if (handler.wrapper) {
            wrappers.push(handler.wrapper);
          }
          //if it's just a function then just register the function
          if (typeof handler.put === 'function') {
            handlerFunction = handler.put;
          } else if (handler.put.callback) {
            //we now expect an object with keys of callback AND / OR wrap
            //callback: fn
            //wrap?: [wrapper, ...]
            handlerFunction = handler.put.callback;
            wrappers = wrappers.concat(handler.put.wrap || []);
          }
          //finally just apply all the above
          //it does not matter if wrappers is an empty array, it will then just be ignored by express
          app.put(route, wrappers, handlerFunction);
        }

        if (handler.delete) {
          // logger.info(`routing_key DELETE${routing_key}`);
          let wrappers = [];
          let handlerFunction = null;
          //apply global wrapper if it exists
          if (handler.wrapper) {
            wrappers.push(handler.wrapper);
          }
          //if it's just a function then just register the function
          if (typeof handler.delete === 'function') {
            handlerFunction = handler.delete;
          } else if (handler.delete.callback) {
            //we now expect an object with keys of callback AND / OR wrap
            //callback: fn
            //wrap?: [wrapper, ...]
            handlerFunction = handler.delete.callback;
            wrappers = wrappers.concat(handler.delete.wrap || []);
          }
          //finally just apply all the above
          //it does not matter if wrappers is an empty array, it will then just be ignored by express
          app.delete(route, wrappers, handlerFunction);
        }
      }
      resolve();
    });
  }

  //NEW SHIT
  function initializeTarly() {
    return TarlyController.initialize(app).setupModels();
  }

  function setupTarlyModels() {
    return TarlyController.setupHandlers().catch(err => {
      logger.error('failed to setupTarlyModels', err, err && err.stack);
      return Promise.resolve();
    });
  }
  //END NEW SHIT

  function logEvents(req, res, next) {
    var code = res.code || 200;
    let elapsed_ms = NaN;
    if (req._audit_event) {
      let ended = new Date();
      elapsed_ms = ended.getTime() - req._audit_event.event_at + 'ms';
    }
    let username = '-';
    let user_id = '-';
    let is_admin = '';
    if (req.acting_user) {
      if (req.acting_user.username) {
        username = req.acting_user.username;
      }
      user_id = req.acting_user.user_id;
    }
    if (req.is_admin) {
      is_admin = 'admin';
    }
    if (code < 500) {
      logger.info(`${req.method} ${req._event.path} ${code} ${elapsed_ms} ${username} ${user_id}`);
    } else {
      logger.error(`${req.method} ${req._event.path} ${code} ${elapsed_ms} ${username} ${user_id}`);
    }
    next();
  }

  function commonResponse(req, res, next) {
    if (!res.code && !req.route && !res.result) {
      res.code = 404;
    }

    if (!res.result) {
      res.result = [];
    }

    if (!Array.isArray(res.result)) {
      res.result = [res.result];
    }

    if (res.redirect_url) {
      res.code = 302;
    }

    if (res.perm_redirect_url) {
      res.code = 301;
    }

    var data = {
      code: Number(res.code) || 200,
      event_id: req._event.id,
      total_results: res.total_results || res.count || res.result.length,
      result: res.result,
      offset: parseInt(req.query.offset),
      count: parseInt(req.query.count),
      current_at: Date.now()
    };

    if (res.merged) {
      for (var key in res.merged) {
        data[key] = res.merged[key];
      }
    }
    if (data.code > 300 && data.code < 400 && res.redirect) {
      if (data.code === 301) {
        res.redirect(res.perm_redirect_url);
      } else {
        res.redirect(res.redirect_url);
      }
      next();
    } else {
      res.status(data.code).json(data);
      next();
    }
  }

  function catchErrors(err, req, res, next) {
    var message = 'Internal Server Error';
    var code = res.code || err.code || 500;
    if (code < 500 || req.is_admin || config.env !== 'bebo-prod') {
      message = err.message || message;
    } else {
      message = http.STATUS_CODES[code];
    }
    var data = {
      code: Number(code),
      event_id: req._event.id,
      message: message
    };
    res.error_message = message;

    res.status(code).json(data);
    if (code == 401) {
      logger.info(req.method + ' ' + req._event.path + ' ' + code, req.headers);
    } else if (code >= 400 && code < 500) {
      logger.info(req.method + ' ' + req._event.path + ' ' + code);
    }
    if (code >= 500) {
      logger.error(req.method + ' ' + req._event.path + ' ' + code);
      logger.error(message, err.stack);
      logger.error('headers', req.headers);
      logger.error('query', req.query);
      logger.error('cookies', req.cookies);
      logger.error('event: ' + JSON.stringify(req._event));
    }
    if (code === 500) {
      Sentry.captureException(err);
    }
    next();
  }

  function apiAudit(req, res, next) {
    return analytics.apiAudit(req, res, next);
  }

  async function init() {
    app.param_black_list = new Set(['bran', 'fortwatch', 'walle', 'pc']);
    app.blocked_route = new Set([]);
    app.param('id', (req, res, next, id) => {
      if (app.param_black_list.has(id)) {
        return next('route');
      }
      return next();
    });

    app.isReservedRoute = route => {
      route = route.replace(/^\/|\/$/g, '');
      return app.blocked_route.has(route);
    };

    app.reserveRoute = route => {
      route = route.replace(/^\/|\/$/g, '');
      let tokens = route.split('/');
      for (let t of tokens) {
        app.param_black_list.add(t);
      }
      return app.blocked_route.add(route);
    };

    app.isValidSubRoute = token => {
      return app.param_black_list.has(token);
    };

    //app.use(forceSSL);
    app.use(Sentry.Handlers.requestHandler());
    app.set('trust proxy', true);
    app.use(setIp);
    app.use(timeout(60000));
    app.use(Bot);
    app.use(setupEvent);
    app.use(cookieParser('dfsvsadf'));
    // app.use(xhub({ algorithm: 'sha1', secret: config.GITHUB_WEBHOOK_SECRET }));

    app.use(busboy());
    app.use(bodyParser.json({ limit: '50mb' })); // to support JSON-encoded bodies (contacts)
    app.use(bodyParser.urlencoded({ limit: '50mb' })); // to support text-encoded bodies (twilio callback)

    var corsOptions = {
      exposedHeaders: ['X-Riano-User-Agent', 'X-Api-Token', 'X-Access-Token', 'X-Riano-User-Id'],
      methods: ['GET', 'PUT', 'POST', 'DELETE', 'OPTIONS'],
      // origin: /http(s)?:\/\/([a-zA-Z0-9-]+\.)?(bebo(-dev|-ptr)?.com|localhost)(:((8(45|55)?0)|(3001|3000)))?$/,
      origin: '*',
      credentials: true
    };

    app.use(cors(corsOptions));
    app.use(auth);
    app.use(queryToSQLOptionsMiddleware);
    setupActiveHandler();
    await initializeTarly();
    await setupHandlers();
    await setupTarlyModels();
    logger.info('All Handlers Registered!!!');
    app.use(commonResponse);
    app.use(emitEvent);
    app.use(logEvents);
    app.use(catchErrors);
    app.use(apiAudit);
    app.use(function(req, res, next) {
      if (!req.timedout) next();
    });
    logger.info('Middleware Added!!');
  }
  await init();
  User.init();
  logger.info('App init done');
  return app;
}

module.exports = App;
