const moment = require('moment');
const Sentry = require('@sentry/node');
const logger = require('../api/logger');
const PharahClient = require('./pharah_client');
const analytics = require('./analytics.js');
const config = require('../config.js');
const cache = require('bebo-node-commons').RedisCache;
const redisClient = cache.getWriteClient(config.TINDER_REDIS_DB);
const _ = require('lodash');
const uuidv4 = require('uuid/v4');
const Redlock = require('redlock');
const { SQS } = require('bebo-node-commons');
const intervalLock =  new Redlock(
  [redisClient],
  {
    driftFactor: 0.01,
    retryCount: 0,
    retryDelay: 500,
    retryJitter: 200
  }
);

const retryLock = new Redlock([redisClient], {
  driftFactor: 0.01,
  retryCount: 1,
  retryDelay: 300,
  retryJitter: 500
});

const SET_START_TIMEOUT = [60, 'm'];
const SET_END_TIMEOUT = [60, 'm'];

const T_SET_START_TIMEOUT = [30, 'm'];
const T_SET_END_TIMEOUT = [30, 'm'];

const LOCK_TTL = 10 * 1000;

const RELEVANT_URLS = [
  '/match',
  '/match/cancel',
  '/match/forfeit',
  '/user/state/bran',
  '/user/state/fortwatch',
  '/user/state/mixer',
  '/user/state/walle',
  '/user/state/pc',
  '/user/state/ready',
  '/user/kill',
  '/user/victory',
  '/user/game',
  '/user/gamestate'
];

function sum(acc, cur) {
  if (cur == null) {
    cur = 0;
  }
  if (acc == null) {
    acc = 0;
  }
  return acc + cur;
}

function guessNowOrSetEndedDttm(set, game_cnt, user_games) {
  // returns now or the maximum set end dttm, whichever is smaller
  if (game_cnt === -1) {
    if (user_games) {
      game_cnt = user_games.length;
    } else {
      game_cnt = set.scores.length + 1;
    }
  }
  let fortniteMinutes = game_cnt * 30;
  logger.debug("guessNowOrSetEndedDttm", game_cnt, fortniteMinutes);
  let now_dttm = new Date();
  let started_dttm = set.started_dttm || set.created_dttm;
  let maxFortnite = moment(started_dttm)
    .add(fortniteMinutes, 'm')
    .toDate();
  now_dttm = new Date(Math.min(now_dttm, maxFortnite));

  // if (user_games && user_games[game_cnt-1]) {
  //   logger.debug("guess ENDED has UG", JSON.stringify(user_games[game_cnt-1]));
  //   logger.debug("guess ENDED has UG", JSON.stringify(set.team));
  // }
  if (
    user_games &&
    set.team &&
    set.team.users &&
    user_games[game_cnt - 1] &&
    user_games[game_cnt - 1].length === set.team.users.length
  ) {
    let last_game_ended = user_games[game_cnt - 1].reduce((accumulator, ug) => {
      if (ug.ended_dttm == null) {
        logger.error('game should be ended', ug.user_game_id, ug);
        return accumulator;
      }
      return Math.max(accumulator, new Date(ug.ended_dttm));
    }, 0);
    now_dttm = Math.min(now_dttm, new Date(last_game_ended));
    // logger.debug("LAST GAME ENDED", last_game_ended.toISOString());
  }
  now_dttm = new Date(now_dttm).toISOString();
  // logger.debug("SET ENDED", new Date(now_dttm).toISOString());
  return now_dttm;
}

const getMatchIdFromMessage = async msg => {
  let match_id = null;
  if (msg.result[0].match_id) {
    match_id = msg.result[0].match_id;
  } else if (msg.user_id) {
    const res = await PharahClient.get('/user', { user_id: msg.user_id });
    match_id = res.result[0].active_match_id;

    if (!match_id) {
      return;
    }
  }

  return match_id;
};

const getMatchByMatchId = async match_id => {
  const res = await PharahClient.get('/match', { match_id });
  const match = res.result[0];

  return match;
};

const getUserGamesForSet = async (set, game_cnt) => {
  const user_games = [];
  const now_dttm = guessNowOrSetEndedDttm(set, game_cnt);

  for (const user of set.team.users) {
    let where = {
      owner_id: user.user_id,
      started_dttm: {
        $gte: set.started_dttm,
        $lte: now_dttm
      }
    };
    let order = [['started_dttm', 'ASC']];
    order = JSON.stringify(order);
    where = JSON.stringify(where);
    let count = game_cnt;
    if (count === -1) {
      count = 100;
    }
    const games = await PharahClient.get('/user/game', {
      where,
      order,
      count
    });

    for (let i = 0; i < games.result.length; i++) {
      if (user_games[i] == null) {
        user_games[i] = [];
      }
      user_games[i].push(games.result[i]);
    }
  }

  return user_games;
};

const timeoutGames = async () => {
  const response = await PharahClient.post('/user/game/maint', {});
  if (!response || response.code != 200) {
    logger.error('Invalid response for /user/game/maint', response);
    return;
  }
};

// TODO - timeout sets

function randomIntFromInterval(min, max) {
  return Math.floor(Math.random() * (max - min + 1) + min);
}


class Storm {

  constructor(matchWorker) {
    this.matchWorker = matchWorker;
  }

  async finalizeMatch(match) {
    match.state = 'ended';
    match.round =  match.sets[0].scores.length;
    await PharahClient.put('/match', {
      match_id: match.match_id,
      state: match.state,
      tournament_score: match.tournament_score,
      round: match.round + 1
    });
  }

  async moveStorm(tournament_id, tournament_round, match_id) {
    await PharahClient.post('/tournament/storm/move', {
      tournament_id: tournament_id,
      round: tournament_round,
      match_id: match_id
    });
  }
}

class HeadsUp {

  constructor(matchWorker) {
    this.matchWorker = matchWorker;
  }

  async finalizeMatch(match, set) {
    let p1_scores = 0;
    let p2_scores = 0;
    for (let game = 0; game < match.game_cnt; game++) {
      p1_scores += match.sets[0].scores[game] || 0;
      p2_scores += match.sets[1].scores[game] || 0;
    }

    if (p1_scores > p2_scores) {
      match.sets[0].outcome = 'win';
      match.sets[1].outcome = 'loss';
    } else if (p1_scores < p2_scores) {
      match.sets[0].outcome = 'loss';
      match.sets[1].outcome = 'win';
    } else {
      if (match.tournament_id) {
        match.sets[0].outcome = 'win';
        match.sets[1].outcome = 'loss';
      } else {
        match.sets[0].outcome = 'draw';
        match.sets[1].outcome = 'draw';
      }
    }

    logger.info('Match result', match.match_id, p1_scores, p2_scores);

    for (const set of match.sets) {
      await PharahClient.put('/set', {
        set_id: set.set_id,
        outcome: set.outcome
      });
    }

    match.state = 'ended';
    await PharahClient.put('/match', {
      match_id: match.match_id,
      state: match.state,
      round: match.game_cnt + 1
    });

    for (const set of match.sets) {
      if (set.team == null) {
        continue;
      }
      for (const user of set.team.users) {
        analytics.track('tinder', 'match', 'ended', null, {
          match_id: match.match_id,
          user_id: user.user_id
        });
      }
    }
    if (match.tournament_id) {
      return;
    }

    try {
      const opponent = match.sets.find(s => s.set_id !== set.set_id);

      let usernames = set.team.users.map(u => u.username);
      usernames = _.join(usernames, ', ');

      let notification = {
        title: 'Check your match score!',
        message: `${usernames} competed against you.`,
        icon_url: opponent.team.image_url,
        require_interaction: true,
        match_id: match.match_id
      };

      const user_ids = opponent.team.users.map(u => u.user_id);
      const campaign_name = 'check_your_match_score';

      PharahClient.post('/user/notification', { notification, user_ids, campaign_name });
    } catch (err) {
      logger.error('ERROR', err, err.stack);
    }
  }
}

class MatchWorker {
  constructor() {
    this.subscribeUrls = RELEVANT_URLS;
    this.onInterval = this.onInterval.bind(this);
    this.onMessage = this.onMessage.bind(this);
    this.interval = randomIntFromInterval(50, 70) * 1000;
    this.HeadsUp = new HeadsUp(this);
    this.Storm = new Storm(this);
  }

  async onInterval() {
    let lock = null;
    try {
      lock = await intervalLock.lock('match_timeout', LOCK_TTL);
      await timeoutGames();
      await this.timeoutSets();
    } catch (error) {
      if (error.name === 'LockError') {
        logger.debug('Failed to get lock match_timout');
      } else {
        logger.error('Interval Error', error);
        Sentry.captureException(error);
      }
    } finally {
      if (lock) {
        await lock.unlock();
      }
    }
  }

  async timeoutSets() {
    logger.debug('timeoutSets');
    const opts = { count: 100 };
    const matchSetRes = await PharahClient.get('/match/expired', opts);
    for (const matchSet of matchSetRes.result) {
      const { match_id, set_id } = matchSet;
      const msg = {
        url: '/set/expire',
        result: [
          {
            url: '/set/expire',
            match_id,
            set_id
          }
        ]
      };
      // logger.debug(msg, JSON.stringify(msg, null, 2));
      await this.onMessage(msg);
    }
  }

  async onMessage(msg) {
    let lock = null;
    let match;
    let matchFunc;

    let match_id = '';
    let event_id = msg.event_id || uuidv4();
    let startTime = process.hrtime();
    let lockTime = null;
    let lock_ms = 0;

    try {
      if (msg.url === '/match') {
        matchFunc = this.onMatch.bind(this);
      } else if (msg.url === '/user/kill') {
        matchFunc = this.onUserKill.bind(this);
      } else if (msg.url === '/user/victory') {
        matchFunc = this.onUserVictory.bind(this);
      } else if (msg.url === '/user/game') {
        matchFunc = this.onUserGame.bind(this);
      } else if (msg.url === '/user/gamestate') {
        matchFunc = this.onUserGameState.bind(this);
      } else if (msg.url === '/match/forfeit') {
        matchFunc = this.onMatchForfeit.bind(this);
      } else if (msg.url === '/set/expire') {
        matchFunc = this.onSetExpire.bind(this);
      } else if (msg.url === '/match/cancel') {
        matchFunc = this.onMatchCancel.bind(this);
      } else if (
        msg.url === '/user/state/bran' ||
        msg.url === '/user/state/fortwatch' ||
        msg.url === '/user/state/pc' ||
        msg.url === '/user/state/walle' ||
        msg.url === '/user/state/mixer' ||
        msg.url === '/user/state/ready'
      ) {
        matchFunc = this.onUserState.bind(this);
      }

      if (matchFunc) {
        match_id = await getMatchIdFromMessage(msg);

        if (!match_id) {
          logger.debug('onMessage url match but not a real match', msg.url);
          return;
        }

        let startLock = process.hrtime();
        lock = await retryLock.lock(match_id, LOCK_TTL);
        // logger.debug('>>> MatchWorker url, match_id', msg.url, match_id);
        lockTime = process.hrtime(startLock);

        match = await getMatchByMatchId(match_id);

        if (!match || match.deleted_dttm) {
          logger.warning('getMatchByMatchId not found or deleted', match);
          return;
        }

        await matchFunc(msg, match);
      }
    } catch (error) {
      if (error.name === 'LockError') {
        logger.info('Failed to get lock - retry later', match_id, msg.url, event_id);
        throw new SQS.RetryError(); // try again later
      } else {
        logger.error('Message processing error', msg, match, event_id);
        logger.error(error);
        Sentry.captureException(error);
      }
      analytics.writeEvent({
        routing_key: 'tinder.error',
        error: `${error}`,
        ...this.getState(match)
      });
    } finally {
      if (lock) {
        await lock.unlock();
      }
      let diff = process.hrtime(startTime);
      let elapsed_ms = (diff[0] * 1e9 + diff[1]) / 1e6;
      if (lockTime) {
        lock_ms = (lockTime[0] * 1e9 + lockTime[1]) / 1e6;
      }
      logger.info('>>>', msg.url, match_id, "[", msg.user_id || "-", "]", event_id, elapsed_ms, "/", elapsed_ms - lock_ms,  'ms');
    }
  }

  getSet(user_id, match) {
    const set = match.sets.find(set => {
      return set.team.users.some(user => user.user_id === user_id);
    });

    if (!set) {
      throw new Error(`Set not found for ${user_id}`);
    }
    return set;
  }

  async onMatch(msg, match) {
    for (let set of match.sets) {
      if (set.state === 'created') {
        logger.info('Match set in created state', match.match_id, set.set_id);

        const userStates = Object.assign({}, (match.data && match.data.state) || {});
        for (const user of set.team.users) {
          const user_id = user.user_id;
          const sBran = await PharahClient.get('/user/state/bran', { user_id });
          const sFortwatch = await PharahClient.get('/user/state/fortwatch', { user_id });
          const sWalle = await PharahClient.get('/user/state/walle', { user_id });
          const sMixer = await PharahClient.get('/user/state/mixer', { user_id });
          const sPc = await PharahClient.get('/user/state/pc', { user_id });
          userStates[user.user_id] = {
            bran: sBran.result[0].exist,
            fortwatch: sFortwatch.result[0].exist,
            gamestate: sFortwatch.result[0].gamestate,
            mode: sFortwatch.result[0].mode,
            walle: sWalle.result[0].exist,
            mixer: sMixer.result[0].exist,
            pc: sPc.result[0].exist,
            ready: false
          };

          analytics.track('tinder', 'match', 'new', null, {
            match_id: match.match_id,
            set_id: set.set_id,
            user_id: user.user_id
          });

          analytics.track('tinder', 'set', 'new', null, {
            match_id: match.match_id,
            set_id: set.set_id,
            user_id: user.user_id
          });
        }

        match.data = { state: userStates };

        set.state = 'waiting';
        await PharahClient.put('/set', {
          set_id: set.set_id,
          state: set.state
        });

        analytics.writeEvent({
          routing_key: 'tinder.new_match',
          ...this.getState(match)
        });

        let matchUpdate = {
          match_id: match.match_id,
          state: match.state,
          data: match.data
        };
        if (match.state === 'created') {
          matchUpdate.state = 'waiting';
        }
        await PharahClient.put('/match', matchUpdate);
      } else if (set.state === 'waiting') {
        let allReady = true;
        for (const user of set.team.users) {
          if (
            !match.data.state[user.user_id] ||
            !match.data.state[user.user_id].bran ||
            !match.data.state[user.user_id].ready
          ) {
            allReady = false;
          }
        }

        logger.info('allReady', set.team.name, allReady);
        if (allReady) {
          await PharahClient.put('/set', {
            set_id: set.set_id,
            started_dttm: '$NOW',
            scores: [],
            state: 'started'
          });

          for (const user of set.team.users) {
            analytics.track('tinder', 'set', 'all_ready', null, {
              set_id: set.set_id,
              match_id: match.match_id,
              user_id: user.user_id
            });

            analytics.track('tinder', 'match', 'all_ready', null, {
              set_id: set.set_id,
              match_id: match.match_id,
              user_id: user.user_id
            });
          }

          await this.startMatch(msg, match);
        } else {
          logger.info('Users not ready yet', match.match_id, match.data);
        }
      } else {
        logger.debug('Match state:', match.state, match.match_id);
      }
    }
  }

  async onUserState(msg, match) {
    const userStateObj = msg.result[0];
    const user_id = userStateObj.user_id;
    if (!user_id) {
      logger.warn('onUserState !user_id', msg);
      return;
    }

    if (!match.data || !match.data.state) {
      logger.error('User states are not intiailzied');
      return;
    }

    const userSet = this.getSet(user_id, match);

    if (userSet.state === 'started') {
      if (msg.url === '/user/state/walle') {
        const newState = match.data.state[user_id] || {};
        newState.walle_url = userStateObj.exist ? userStateObj.relay_url : null;
        newState.walle_hls_url = userStateObj.exist ? userStateObj.hls_url : null;
        match.data.state[user_id] = newState;

        await PharahClient.put('/match', {
          match_id: match.match_id,
          data: match.data
        });
      }
    }

    if (userSet.state !== 'waiting') {
      return;
    }

    logger.info('User state while waiting', msg);

    const newState = match.data.state[user_id] || {};

    if (msg.url === '/user/state/bran') {
      newState.bran = userStateObj.exist;
    } else if (msg.url === '/user/state/fortwatch') {
      newState.fortwatch = userStateObj.exist;
      newState.mode = userStateObj.mode;
      newState.gamestate = userStateObj.gamestate;
    } else if (msg.url === '/user/state/walle') {
      newState.walle = userStateObj.exist;
      newState.walle_url = userStateObj.exist ? userStateObj.relay_url : null;
      newState.walle_hls_url = userStateObj.exist ? userStateObj.hls_url : null;
    } else if (msg.url === '/user/state/mixer') {
      newState.mixer = userStateObj.exist;
      newState.mixer_url = userStateObj.exist ? userStateObj.mixer_url : null;
    } else if (msg.url === '/user/state/pc') {
      newState.pc = userStateObj.exist;
    } else if (msg.url === '/user/state/ready') {
      newState.ready = userStateObj.exist;

      analytics.track('tinder', 'match', 'ready', null, {
        match_id: match.match_id,
        set_id: userSet.set_id,
        user_id
      });

      analytics.track('tinder', 'set', 'ready', null, {
        match_id: match.match_id,
        set_id: userSet.set_id,
        user_id
      });
    }

    match.data.state[user_id] = newState;

    await PharahClient.put('/match', {
      match_id: match.match_id,
      data: match.data
    });
  }

  async startMatch(msg, match) {
    logger.info('Starting match', match.match_id);

    let matchUpdate = {
      match_id: match.match_id,
      updated_dttm: '$NOW'
    };
    if (match.state === 'waiting') {
      matchUpdate.state = 'started';
    }
    await PharahClient.put('/match', matchUpdate);

    let originalMatch = null;
    if (!match.tournament_id) {
      const newMatch = await PharahClient.post('/match/finder', {
        match_id: match.match_id
      });

      originalMatch = match;
      match = newMatch.result[0];
    }

    if (match.sets.length === 1) {
      logger.info('Match started without opponent', match.match_id);

      analytics.writeEvent({
        routing_key: 'tinder.no_opponent',
        ...this.getState(match)
      });

      for (const user of match.sets[0].team.users) {
        analytics.track('tinder', 'match', 'no_opponent', null, {
          match_id: match.match_id,
          user_id: user.user_id
        });

        analytics.track('tinder', 'match', 'started', null, {
          match_id: match.match_id,
          user_id: user.user_id
        });

        analytics.track('tinder', 'set', 'started', null, {
          match_id: match.match_id,
          user_id: user.user_id
        });
      }
    } else if (match.sets.length === 2) {
      logger.info('Match started with opponent', match.match_id, match.sets);

      analytics.writeEvent({
        routing_key: 'tinder.found_opponent',
        ...this.getState(match)
      });

      for (const set of match.sets) {
        for (const user of set.team.users) {
          analytics.track('tinder', 'match', 'found_opponent', null, {
            match_id: match.match_id,
            user_id: user.user_id
          });
        }
      }

      if (originalMatch) {
        // original team whos now placed in a waiting match
        for (const user of originalMatch.sets[0].team.users) {
          analytics.track('tinder', 'match', 'started', null, {
            match_id: match.match_id,
            user_id: user.user_id
          });

          analytics.track('tinder', 'set', 'started', null, {
            match_id: match.match_id,
            user_id: user.user_id
          });
        }
      }
    } else {
      throw new Error('wtf wrong set length');
    }
  }

  async onMatchForfeit(msg, match) {

    // TODO should probably take time from api reponse
    logger.debug('onMatchForfeit', msg);

    const result = msg.result[0];

    let set = this.getSet(result.user_id, match);
    let user_games = await getUserGamesForSet(set, match.game_cnt);
    let now_dttm = result.ended_dttm;
    if (now_dttm == null) {
      now_dttm = guessNowOrSetEndedDttm(set, match.game_cnt);
    }

    for (const ug of _.flatten(user_games)) {
      if (ug.ended_dttm == null) {
        ug.ended_dttm = now_dttm;
        const update = {
          user_game_id: ug.user_game_id,
          ended_dttm: now_dttm
        };
        const response = await PharahClient.put('/user/game', update);
        if (!response || response.code != 200) {
          logger.error('Invalid response for PUT /user/game', response);
        }
      }
    }
    return await this.doMatchForfeit(match, set, user_games);
  }

  async doMatchForfeit(match, set, user_games) {
    analytics.writeEvent({
      routing_key: 'tinder.forfeit',
      ...this.getState(match)
    });

    let endedRounds = this.getEndedRounds(user_games);
    if (endedRounds === 0) {
      logger.warning('forfeit with 0 games?', JSON.stringify(user_games));
    } else {
      for (let i = 0; i < endedRounds; i++) {
        await this.finalizeGame(set, user_games[i], i + 1, match);
      }
    }

    if (set.state !== 'ended') {
      logger.info(
        set.set_id,
        'set.scores.length',
        set.scores.length,
        'match.game_cnt',
        match.game_cnt
      );
      let now_dttm = guessNowOrSetEndedDttm(set, match.game_cnt, user_games);
      await this.finalizeSet(set, match, true, now_dttm);
    }
  }

  writeExpireEvent(match, set, late, action) {
    analytics.writeEvent({
      routing_key: 'tinder.set.expire',
      late_dec: late,
      next_tx: action,
      ...this.getState(match, set)
    });
  }

  async onSetExpire(msg, match) {
    // TODO should probably take time from api reponse
    logger.debug('onSetExpire', msg);
    const result = msg.result[0];
    const set_id = result.set_id;
    const set = match.sets.find(s => s.set_id == set_id);
    let now_dttm = new Date();
    let start_timeout = SET_START_TIMEOUT;
    let end_timeout = SET_END_TIMEOUT;
    if (match.tournament_id) {
      end_timeout = T_SET_END_TIMEOUT;
      start_timeout = T_SET_START_TIMEOUT;
    }

    if (!set.started_dttm) {
      let late =
        (moment(now_dttm)
          .subtract(...start_timeout)
          .toDate() -
          new Date(set.created_dttm)) /
        1000;
      if (
        moment(set.created_dttm)
          .add(...start_timeout)
          .isBefore(moment())
      ) {
        if (match.tournament_id) {
          logger.error(
            'match_id',
            match.match_id,
            'set_id',
            set_id,
            'late (seconds)',
            late,
            'this.doMatchForfeit(match)'
          );
          this.writeExpireEvent(match, set, late, 'forfeit');
          return this.doMatchForfeit(match, set, []);
        } else {
          logger.error(
            'match_id',
            match.match_id,
            'set_id',
            set_id,
            'late (seconds)',
            late,
            'this.doMatchCancel(match)'
          );
          this.writeExpireEvent(match, set, late, 'cancel');
          return this.doMatchCancel(match);
        }
      } else {
        logger.debug(
          'match_id',
          match.match_id,
          'set_id',
          set_id,
          'late (seconds)',
          late,
          'still time to get ready'
        );
        return;
      }
    }

    let user_games = await getUserGamesForSet(set, match.game_cnt);
    let all_games = _.flattenDeep(user_games);
    let is_playing = all_games.some(ug => ug.ended_dttm == null);

    let last_game_ended = all_games.reduce((accumulator, ug) => {
      return Math.max(accumulator, new Date(ug.ended_dttm));
    }, new Date(set.started_dttm));
    last_game_ended = new Date(last_game_ended);

    if (is_playing) {
      logger.debug('match_id', match.match_id, 'set_id', set_id, 'still playing match_id');
      return;
    }

    let late =
      (moment(now_dttm)
        .subtract(...end_timeout)
        .toDate() -
        new Date(last_game_ended)) /
      1000;
    if (
      last_game_ended >
      moment(now_dttm)
        .subtract(...end_timeout)
        .toDate()
    ) {
      logger.debug(
        'match_id',
        match.match_id,
        'set_id',
        set_id,
        'late (seconds)',
        late,
        'still in waiting period',
        'last_game_ended',
        last_game_ended
      );
      return;
    }

    if (all_games.length === 0 && match.tournament_id === null && match.sets.length === 1) {
      logger.info(
        'match_id',
        match.match_id,
        'set_id',
        set.set_id,
        'late (seconds)',
        late,
        'expiring match -> cancel',
        'number of games',
        all_games.length,
        'tournament_id',
        match.tournament_id,
        'sets',
        match.sets.length
      );
      this.writeExpireEvent(match, set, late, 'cancel');
      return this.doMatchCancel(match);
    }
    logger.info(
      'match_id',
      match.match_id,
      'set_id',
      set.set_id,
      'late (seconds)',
      late,
      'expiring match -> forfeit',
      'number of games',
      all_games.length,
      'tournament_id',
      match.tournament_id,
      'sets',
      match.sets.length
    );
    this.writeExpireEvent(match, set, late, 'forfeit');
    return await this.doMatchForfeit(match, set, user_games);
  }

  async leaveMatch(user_id, match_id) {
    logger.debug('leaveMatch', user_id, match_id);
    await PharahClient.delete('/user/active/match', {
      user_id,
      match_id
    });
    // same here - we should probably only make not ready if the
    // same match_id
    await PharahClient.put('/user', {
      user_id,
      current_platform: null
    });
  }

  async onMatchCancel(msg, match) {
    return await this.doMatchCancel(match);
  }

  async doMatchCancel(match) {
    analytics.writeEvent({
      routing_key: 'tinder.cancel',
      ...this.getState(match)
    });

    if (match.sets.length !== 1) {
      logger.error(
        '/match/cancel must only happen before we have an opponent',
        'match_id:',
        match.match_id,
        'number of sets:',
        match.sets.length
      );
      return;
    }

    for (const user of match.sets[0].team.users) {
      await this.leaveMatch(user.user_id, match.match_id);
      analytics.track('tinder', 'match', 'cancel', null, {
        match_id: match.match_id,
        user_id: user.user_id
      });
      // const user_games = await getUserGamesForSet(set);
      // TODO need to find games that haven't been ended yet and end
      // can you cancel once you are in a game? (@fpn)
      // logger.info("user_games", user_games);
    }

    await PharahClient.put('/match', {
      match_id: match.match_id,
      deleted_dttm: '$NOW'
    });
  }

  async onUserGameState(msg, match) {
    const game = msg.result[0];
    const user_id = game.user_id;
    const user = await PharahClient.get('/user', { user_id: user_id });

    if (user.current_platform === 'xbox') {
      this.onUserGame(msg, match);
    }
  }

  getEndedRounds(user_games) {
    let rounds = 0;
    for (const round of user_games) {
      const endedGames = round.filter(g => g.ended_dttm);
      if (endedGames.length !== round.length) {
        break;
      }
      rounds++;
    }
    logger.info('getEndedRounds', rounds);
    return rounds;
  }

  async onUserGame(msg, match) {
    const game = msg.result[0];
    const set = this.getSet(game.user_id, match);

    if (match.state !== 'started' || set.state !== 'started') {
      logger.warn('user game when match or set is not started');
      return;
    }

    if (game.ended_dttm && new Date(game.ended_dttm) < new Date(set.started_dttm)) {
      logger.warn('onUserGame old game ended before set started');
      return;
    }

    const user_games = await getUserGamesForSet(set, match.game_cnt);
    let endedRounds = this.getEndedRounds(user_games);

    if (!game.ended_dttm) {
      // logger.info("onUserGame game started", JSON.stringify(game, null, 2));

      let currentRound = endedRounds + 1;
      if (match.game_cnt != -1 && currentRound > match.game_cnt) {
        logger.warn(
          'starting more games that expected in match',
          match.match_id,
          currentRound,
          '>',
          match.game_cnt
        );
        return;
      }

      analytics.track('user', 'game', 'start', null, {
        match_id: match.match_id,
        user_id: game.user_id,
        round: currentRound
      });

      await this.startGame(set, user_games[currentRound - 1], currentRound, match);
      return;
    }

    // logger.debug("onUserGame game end", JSON.stringify(game, null, 2));

    analytics.track('user', 'game', 'end', null, {
      match_id: match.match_id,
      user_id: game.user_id,
      round: set.round
    });

    if (endedRounds >= set.round) {
      logger.info('calling finalizeGame endedRounds:', endedRounds, 'set.round', set.round);
      if (endedRounds === 0 || set.round === 0) {
        logger.warning('unrecognized game end', game.user_id);
        return;
      }
      await this.finalizeGame(set, user_games[endedRounds - 1], endedRounds, match);
    }
  }

  async startGame(set, user_games, round, match) {
    // logger.info("startGame round, user_games", round, JSON.stringify(user_games, null, 2));

    set.round = round;
    const setUpdate = {
      set_id: set.set_id,
      round: set.round
    };

    if (!set.scores[round - 1]) {
      set.scores[round - 1] = 0;
      setUpdate.scores = set.scores;
    }

    await PharahClient.put('/set', setUpdate);

    if (match.sets.length === 2) {
      match.round = Math.max(Math.min(match.sets[0].round, match.sets[1].round) - 1, 0);
    }

    let matchUpdate = {
      match_id: match.match_id,
      round: match.round,
      updated_dttm: '$NOW'
    };
    await PharahClient.put('/match', matchUpdate);
  }

  async onUserKill(msg, match) {
    const user_kill = msg.result[0];
    const user_id = user_kill.owner_id;
    analytics.track('user', 'game', 'kill', null, {
      match_id: match.match_id,
      user_id,
      round: match.round // FIXME
    });

    const now_dttm = user_kill.created_dttm;
    let set = this.getSet(user_id, match);
    await this.scoreCurrentRound(set, match, now_dttm);
  }

  async onUserVictory(msg, match) {
    const user_victory = msg.result[0];
    const user_id = user_victory.owner_id;
    analytics.track('user', 'game', 'victory', null, {
      match_id: match.match_id,
      user_id,
      round: match.round // FIXME
    });
    const now_dttm = user_victory.created_dttm;
    let set = this.getSet(user_id, match);
    await this.scoreCurrentRound(set, match, now_dttm);
  }

  async scoreCurrentRound(set, match, now_dttm, advanceRound) {
    let old_score = set.scores.reduce(sum,0);
    let user_games = await getUserGamesForSet(set, match.game_cnt);
    let user_round_games = user_games[set.round - 1];
    if (!user_round_games) {
      logger.info('no game to update', set.round, user_games);
      return;
    }
    for (const ug of user_round_games) {
      if (ug.ended_dttm == null) {
        ug.ended_dttm = now_dttm;
      }
    }
    await this.scoreRound(set, user_round_games, set.round, match);
    let new_score = set.scores.reduce(sum);
    if (advanceRound) {
      set.round += 1;
    }
    const setUpdate = {
      set_id: set.set_id,
      scores: set.scores,
      round: set.round
    };
    await PharahClient.put('/set', setUpdate);
    // let tournament_seed = Math.floor((moment()-moment(match.created_dttm))/100);
    let matchUpdate = {
      match_id: match.match_id,
      tournament_score: new_score,
      // tournament_seed,
      updated_dttm: '$NOW'
    };
    await PharahClient.put('/match', matchUpdate);
    if (new_score != old_score) {
      if (match.tournament_type === "storm") {
        await this.Storm.moveStorm(match.tournament_id, match.tournament_round, match.match_id);
      }
    }
  }

  async scoreRound(set, user_games, round, match) {
    let totalPoints = 0;
    let victory = false;
    // TODO is this needed?
    const now_dttm = guessNowOrSetEndedDttm(set, match.game_cnt);

    // logger.info("scoreRound", set.set_id, round, JSON.stringify(user_games, null, 2));
    for (const user_game of user_games) {
      let where = {
        owner_id: user_game.user_id,
        created_dttm: {
          $gte: user_game.started_dttm,
          $lte: user_game.ended_dttm || now_dttm
        }
      };
      where = JSON.stringify(where);
      let opts = { where, count: 100 };

      const kills = await PharahClient.get('/user/kill', opts);
      // logger.info("KILLS", set.set_id, user_game.user_id,  JSON.stringify(kills, null, 2));
      const victories = await PharahClient.get('/user/victory', opts);

      totalPoints += kills.total_results;
      victory = victory || victories.total_results !== 0;
    }

    if (victory) {
      totalPoints += 10;
    }

    logger.info(
      'score update',
      match.match_id,
      'set_id',
      set.set_id,
      set.round,
      '->',
      round,
      totalPoints,
      victory
    );

    set.scores[round - 1] = totalPoints;
  }

  async finalizeGame(set, user_games, round, match) {
    logger.info('finalizeGame', set.set_id, round);
    const endedGames = user_games.filter(g => g.ended_dttm);
    if (endedGames.length !== user_games.length) {
      logger.error(
        'Game should only be finalized if everyone is done',
        endedGames,
        '!=',
        user_games.length,
        'players',
        set.team && set.team.users.length,
        'round',
        round,
        JSON.stringify(user_games, null, 2)
      );
      return;
    }

    analytics.writeEvent({
      routing_key: 'tinder.game_end',
      ...this.getState(match)
    });

    for (const user of set.team.users) {
      analytics.track('tinder', 'match', 'game_end', null, {
        match_id: match.match_id,
        user_id: user.user_id
      });
    }

    let now_dttm = guessNowOrSetEndedDttm(set, match.game_cnt, user_games);

    if (round === match.game_cnt) {
      await this.finalizeSet(set, match, false, now_dttm);
    } else {
      let advanceRound = true;
      this.scoreCurrentRound(set, match, now_dttm, advanceRound);
    }
  }

  async finalizeSet(set, match, forfeit, now_dttm) {
    logger.info('Finalize set', match.match_id, forfeit, match.game_cnt);
    let old_score = set.scores.reduce(sum, 0);
    let user_games = await getUserGamesForSet(set, match.game_cnt);
    let user_round_games = user_games[set.round-1]
    if (!user_round_games) {
      logger.info('no game to update', set.round, user_games);
    } else {
      await this.scoreRound(set, user_round_games, set.round, match);
    }
    let new_score = set.scores.reduce(sum, 0);

    set.state = 'ended';
    if (match.game_cnt !== -1) {
      set.round = match.game_cnt + 1;
    } else {
      set.round += 1;
    }

    while (set.scores.length < match.game_cnt) {
      set.scores.push(0);
    }

    await PharahClient.put('/set', {
      set_id: set.set_id,
      round: set.round,
      state: set.state,
      scores: set.scores,
      ended_dttm: now_dttm
    });

    analytics.writeEvent({
      routing_key: forfeit ? 'tinder.forfeit' : 'tinder.set_end',
      ...this.getState(match)
    });

    // logger.debug("should leave match",
    //   JSON.stringify(set.team.users, null, 2));
    for (const user of set.team.users) {
      let action = forfeit ? 'forfeit' : 'end';
      analytics.track('tinder', 'set', action, null, {
        match_id: match.match_id,
        user_id: user.user_id
      });

      await this.leaveMatch(user.user_id, match.match_id);
      //const user_games = await getUserGamesForSet(set, match.game_cnt);

      ////TODO need to find games that haven't been ended yet and end
      //logger.info("user_games", user_games);
    }

    PharahClient.put('/team', {
      team_id: set.team_id,
      last_match_dttm: '$NOW'
    });

    if (match.sets.filter(set => set.state === 'ended').length < match.sets.length) {
      set.outcome = 'waiting';
      await PharahClient.put('/set', {
        set_id: set.set_id,
        outcome: set.outcome
      });

      let matchUpdate = {
        match_id: match.match_id,
        tournament_score: new_score,
        updated_dttm: '$NOW'
      };
      await PharahClient.put('/match', matchUpdate);
    } else if (match.sets.filter(set => set.state === 'ended').length === match.sets.length) {
      match.tournament_score = new_score;
      await this.finalizeMatch(match, set);
    } else {
      throw new Error('wtf wrong set length');
    }
  }

  async finalizeMatch(match, set) {
    if (match.tournament_type === "storm") {
      this.Storm.finalizeMatch(match);
    } else {
      this.HeadsUp.finalizeMatch(match, set);
    }
  }

  getState(match, set) {
    if (match == null) {
      return;
    }
    try {
      let match_event = {
        match_id: match.match_id,
        match: {
          raw: JSON.stringify(match),
          league_id: match.league_id,
          data_tx: JSON.stringify(match.data) || ''
        }
      };
      const m = match_event.match;

      if (set && set.set_id) {
        match_event.set_id = set.set_id;
      }

      if (match.sets && match.sets[0]) {
        if (match.sets[0].team) {
          m.challenger_usernames = match.sets[0].team.users.map(u => u.username).join(',') || '';
        }
        m.challenger_scores_tx = JSON.stringify(match.sets[0].scores) || '';
        m.challenger_score_nr = match.sets[0].scores.reduce((a, b) => a + b, 0) || 0;
      }
      if (match.sets && match.sets[1]) {
        if (match.sets[1].team) {
          m.opponent_usernames =
            (match.sets[1] && match.sets[1].team.users.map(u => u.username).join(',')) || '';
        }
        m.opponent_scores_tx = JSON.stringify((match.sets[1] || {}).scores) || '';
        m.opponent_score_nr = match.sets[1].scores.reduce((a, b) => a + b, 0) || 0;
      }
      return match_event;
    } catch (err) {
      logger.error('fail to MatchWorker.getState', err);
      return {
        match: {
          raw_tx: JSON.stringify(match)
        }
      };
    }
  }
}

const singleton = new MatchWorker();
module.exports = singleton;
