const moment = require('moment');
const logger = require('../api/logger');
const PharahClient = require('./pharah_client');
const analytics = require('./analytics.js');
const _ = require('lodash');

const ONE_HOUR_IN_MS = 3600 * 1000;

const getTournament = async (tournament_id) => {
  const tournamentRes = await PharahClient.get("/tournament", {tournament_id});
  return tournamentRes.result[0];
};


const getTournamentLeagues = async () => {
  const leagueRes = await PharahClient.get("/league", {count: 100, where: JSON.stringify({active_tournament_id: {"$ne": null}}) });
  return leagueRes.result;
};

const randomIntFromInterval = (min,max) => {
  return Math.floor(Math.random() * (max-min+1)+min);
};


class BaseTournamentWorker {

  constructor() {
    this.interval = randomIntFromInterval(3000, 10000);
    this.running = false;
    this.supportedTypes = [];
    logger.info("this.interval", this.interval);
  }

  async getStartedTournaments() {
    // TODO add sorting
    const where = JSON.stringify({
      state:"started", 
      type: {"$in": this.supportedTypes}});
    const tournamentResult = await PharahClient.get("/tournament", {where, count: 100});
    return tournamentResult.result;
  }

  async getTournamentTeams(tournament_id) {
    const where = JSON.stringify({
      tournament_id, 
      checkin_dttm: {"$ne": null}});
    const teams = await PharahClient.get('/tournament/team', {where, count: 257});
    const tournament_teams = teams.result;
    return tournament_teams;
  }

  async setTournamentsToStarted() {
    const where = JSON.stringify({
      state:"created", 
      type: {"$in": this.supportedTypes}});
    const tournamentResult = await PharahClient.get("/tournament", {count:100, where});
    for(const tournament of tournamentResult.result){
      let {start_dttm, state, tournament_id, league_id, points_needed} = tournament;


      let first_team = await PharahClient.get("/tournament/team", {
        tournament_id, 
        count: 1});
      let checkin = false;
      if (first_team && first_team.result && first_team.result.length > 0) {
        checkin = true;
      }

      if (!checkin && state === "created" && (Date.now() + ONE_HOUR_IN_MS) > new Date(start_dttm)){

        this.track("tournament.start.checkin", {tournament_id, league_id: league_id});

        const leagueRes = await PharahClient.get('/league', {league_id});
        const league = leagueRes.result[0];

        logger.info(tournament_id, "created -> checkin");
        const leagueQualified = await PharahClient.get("/league/qualified", {league_id, count:2048});
        const winnerStandings = leagueQualified.result.filter(s => parseInt(s.points) >= points_needed);

        for (const team of winnerStandings){
          await PharahClient.post("/tournament/team", {
            tournament_id, 
            team_id: team.team.team_id, 
            rank: parseInt(team.win),
          });
          this.track("tournament.create.team", {tournament_id, league_id: league.league_id});
        }

        try {
          const checkin_notification =  {
            title: `Tournament check-in required`,
            message: `You're qualified for the ${tournament.name} in ${league.name}`, 
            icon_url: league.image_url,
            require_interaction: true,
            tournament_id: tournament.tournament_id
          };
          const users = _.flatten(winnerStandings.map(s => s.team.users));
          const user_ids = users.map(u => u.user_id);
          const campaign_name = 'tournament_checkin';
          PharahClient.post('/user/notification', { notification: checkin_notification, user_ids, campaign_name });
        } catch (err) {
          logger.error('ERROR SENDING CHECKIN NOTFICATION', err, err.stack);
        }
      }

      if (state === "created" && (Date.now() + 5000) > new Date(start_dttm)){

        const leagueRes = await PharahClient.get('/league', {league_id});
        const league = leagueRes.result[0];

        logger.info(`checkin -> started - league: ${league.league_id}`);

        const where = JSON.stringify({
          tournament_id, 
          checkin_dttm: {"$ne": null}});

        logger.info("setTournamentsToStarted", start_dttm, state, tournament_id, league_id);
        logger.info(`tournament_id: ${tournament_id} created and should be started, ${tournament.name} ${Date.now()} > ${new Date(start_dttm)}`);
        let teams = await PharahClient.get("/tournament/team", { where, count: 257 }) ;

        points_needed = parseInt(points_needed);
        const teamCount = teams.result && teams.result.length;
        if (!teamCount) {
          logger.error("no team ??? tournament_id", tournament_id);
          continue;
        }
        if (teamCount < 2) {
          logger.error("need at least 2 teams for tournament_id", tournament_id);
          continue;
        }
        this.track("tournament.start", {tournament_id, league_id: league.league_id, round: 0, team_cnt: teamCount});

        let maxRounds = this.calculateRounds(tournament, teamCount);
        let firstRound = this.calculateFirstRound(tournament, teamCount);
        tournament.round = firstRound;
        let nextRoundDttm = this.calculateNextRoundDttm(tournament);
        logger.debug(tournament_id, "tournament.start", {round: 0, team_cnt: teamCount, maxRounds, firstRound, nextRoundDttm});

        await PharahClient.put("/tournament", {
          tournament_id, 
          state: "started",
          round: firstRound,
          round_cnt: maxRounds,
          next_round_dttm: nextRoundDttm 
        });
        this.track("tournament.start", {tournament_id, league_id: league.league_id, round: 1, round_cnt: maxRounds});
        await PharahClient.put("/league", {league_id, active_tournament_id: tournament_id});
        this.track("tournament.activate.league", {tournament_id, league_id: league.league_id});
        try {
          const start_notification =  {
            title: `Tournament starts now!`,
            message: `${tournament.name}: First round starting`,
            icon_url: league.image_url,
            require_interaction: true,
            tournament_id
          };
          const users = _.flatten(teams.result.map(t => t.users));
          const user_ids = users.map(u => u.user_id);
          const campaign_name = 'tournament_start';
          PharahClient.post('/user/notification', { notification: start_notification, user_ids, campaign_name });
        } catch (err) {
          logger.error('ERROR SENDING START NOTFICATION', err, err.stack);
        }
      }
    }
  }
  async setTournamentsToEnd() {
    const startedTournaments =  await this.getStartedTournaments();
    startedTournaments.reduce((acc, value) => {
      if(value.league_id in acc){
        acc[value.league_id].push(value);
        return acc;
      } else {
        acc[value.league_id] = [value];
        return acc;
      }
    }, {});

    for(const league_id in startedTournaments){
      if(startedTournaments[league_id].length > 1){
        //TODO double check sort
        const sorted = startedTournaments[league_id].sort((a,b) => new Date(b.start_dttm) - new Date(a.start_dttm));
        sorted.shift();
        for(const t of sorted){
          logger.info(`setting tournament: ${t.tournament_id}: ended now`);
          await PharahClient.put("/tournament", {tournament_id: t.tournament_id, end_dttm: "$NOW", state: "ended" });
          this.track("tournament.end", {tournament_id: t.tournament_id});
        }
      }
    }
  }

  async removeEndedTournamentsFromLeague() {
    const leagues = await getTournamentLeagues();
    // logger.info("removeEndedTournamentsFromLeague leagues", leagues);

    for(const league of leagues){
      // logger.info("league", league);
      const {active_tournament_id: tournament_id, league_id} = league;
      // logger.info("removeEndedTournamentsFromLeague tournament_id, league_id", tournament_id, league_id);
      const tournament = await getTournament(tournament_id);
      // logger.info("removeEndedTournamentsFromLeague tournament", tournament);

      if(tournament.state === "ended" 
        && tournament.end_dttm 
        && new moment(tournament.end_dttm).add(30, 'minutes').isBefore(moment()))  {
        // logger.info("tournamend has ended and timed out, remove from the league");
        await PharahClient.put("/league", {league_id, active_tournament_id: null});
        this.track("tournament.deactivate.league", {tournament_id, league_id});
      }
    }
  }

  async onInterval() {
    logger.debug("override?");
  }

  async onMessage(msg) {
    logger.debug("override?", msg);
  }

  track (routing_key, data={}) {
    const splitRoutingKey = routing_key.split(".");
    const category = splitRoutingKey[0];
    const action = splitRoutingKey[1];
    const label = splitRoutingKey[1] || null;

    const event = Object.assign({}, {routing_key, category, action, label}, data);
    analytics.writeEvent(event);
  }
}

BaseTournamentWorker.TournamentType = {
  SINGLE_ELIM: "single_elim",
  DOUBLE_ELIM: "double_elim",
  STORM: "storm"
};
module.exports = BaseTournamentWorker;
