const Redlock = require('redlock');
const Bracket = require('../classes/bracket');
const _ = require('lodash');
const TournamentWorkerBase = require('./tournament_worker_base');
const TournamentType = TournamentWorkerBase.TournamentType;
const logger = require('../api/logger');
const PharahClient = require('./pharah_client');
const analytics = require('./analytics.js');
const config = require('../config.js');
const cache = require('bebo-node-commons').RedisCache;
const redisClient = cache.getWriteClient(config.TINDER_REDIS_DB);

const intervalLock =  new Redlock(
  [redisClient],
  {
    driftFactor: 0.01,
    retryCount: 0,
    retryDelay: 500,
    retryJitter: 200
  }
);

const LOCK_TTL = 60 * 1000;
const UPPER_GAME_CNT = 2;
const LOWER_GAME_CNT = 2;
const FINAL_GAME_CNT = 2;

const bracketSize = (count) => {
  let p = (count-1).toString(2).length;
  return Math.pow(2, p);
};


const makeBracket = (ranks, lowHigh) => {
  // requires rank sorted list
  let bracket = [];
  let length = ranks.length;
  if(length === 0){
    return [];
  }

  let size = bracketSize(length);
  let byes = size - length; 
  logger.debug("team count", length, "bracket size", size, "byes", byes);
  for (let i = 0 ; i < byes; i++) {
    let a = ranks.shift();
    let match = {
      state: "bye",
      teams: [a]
    };
    bracket.push(match);
  }
  // now we talk lowest and highest
  while(ranks.length > 0) {
    let a = ranks.shift();
    let b = null;
    if (lowHigh) {
      b = ranks.pop();
    } else {
      b = ranks.shift();
    }
    let match = {
      state: "created",
      teams: [a, b]
    };
    bracket.push(match);
  }
  return bracket;
};

const makeLowerBracket = (ranks, lowHigh) => {
  // requires rank sorted list
  let bracket = [];
  let length = ranks.length;
  if(length === 0){
    return [];
  }

  let byes = length % 2;
  logger.debug("lower team count", length, "byes", byes);
  for (let i = 0 ; i < byes; i++) {
    let a = ranks.shift();
    let match = {
      state: "bye",
      teams: [a]
    };
    bracket.push(match);
  }
  // now we take lowest and highest
  while (ranks.length > 0) {
    let a = ranks.shift();
    let b = null;
    if (lowHigh) {
      b = ranks.pop();
    } else {
      b = ranks.shift();
    }
    let match = {
      state: "created",
      teams: [a, b]
    };
    bracket.push(match);
  }
  return bracket;
};

const rankCompare = (a, b) => {
  return b.rank - a.rank;
};

const seedCompare = (a, b) => {
  return a.seed - b.seed;
};

const RELEVANT_URLS = [
  '/tournament'
];

class TournamentWorkerBracket extends TournamentWorkerBase {

  constructor() {
    super();
    this.subscribeUrls = RELEVANT_URLS;
    this.onInterval = this.onInterval.bind(this);
    this.onMessage = this.onMessage.bind(this);
    this.supportedTypes = [TournamentType.SINGLE_ELIM, TournamentType.DOUBLE_ELIM];
  }

  calculateRounds(tournament, teamCount) {

    let maxRounds = 0;
    if (tournament.type === TournamentType.DOUBLE_ELIM) {
      maxRounds = Bracket.Double.rounds(teamCount);
    } else {
      maxRounds = Bracket.Single.rounds(teamCount);
    }
    return maxRounds;
  }

  calculateFirstRound() {
    return 1; 
  }

  calculateNextRoundDttm() {
    return null;
  }

  async createFirstUpperBracket(tournament_teams, tournament_id, round, game_cnt) {

    tournament_teams.sort(rankCompare);
    tournament_teams.map(async (team, i) => {
      team.seed = i+1;
      await PharahClient.put("/tournament/team", {tournament_id, team_id: team.team_id, seed: team.seed});
      this.track("tournament.update.team", {tournament_id, team_id: team.team_id, seed: team.seed});
    });

    logger.info("tournament_teams", tournament_teams);

    let brackets = makeBracket(tournament_teams, true);
    for(const bracket of brackets){
      const team_ids = bracket.teams.map(t => t.team_id);
      const seeds = bracket.teams.map(t => t.seed);
      const tournament_seed = Math.min(...seeds);
      logger.info(`bracket round ${round} team_ids ${team_ids}`);
      await PharahClient.post("/match", 
        { tournament_id,
          tournament_round: round,
          tournament_seed: tournament_seed,
          tournament_bracket: "upper",
          state: bracket.state,
          team_ids: team_ids,
          game_cnt});
      this.track("tournament.create.match", {tournament_id, round: round, state: bracket.state, team_ids: JSON.stringify(team_ids), bracket: "upper", game_cnt});
    }
  }

  async createUpperBracket(tournament_teams, tournament_id, round, game_cnt) {

    tournament_teams.sort(seedCompare); // seed ascending
    // logger.info("tournament_teams", tournament_teams);
    logger.info("upper bracket tournament_teams", tournament_id);
    let brackets = makeBracket(tournament_teams, true);
    for(const bracket of brackets){
      const team_ids = bracket.teams.map(t => t.team_id);
      const seeds = bracket.teams.map(t => t.seed);
      const tournament_seed = Math.min(...seeds);
      logger.info(`bracket round ${round} team_ids ${team_ids}`);
      await PharahClient.post("/match", 
        { tournament_id,
          tournament_round: round,
          tournament_bracket: "upper",
          tournament_seed: tournament_seed,
          state: bracket.state,
          team_ids: team_ids,
          game_cnt});
      this.track("tournament.create.match", {tournament_id, round: round, state: bracket.state, team_ids: JSON.stringify(team_ids), bracket: "upper", game_cnt});
    }
  }

  async createLowerBracket(upper_tournament_losers, lower_tournament_winners, tournament_id, round, game_cnt) {

    upper_tournament_losers.sort(seedCompare); // seed ascending
    let tournament_teams = upper_tournament_losers;

    // let lowHigh = false;
    let lowHigh = true;
    if (lower_tournament_winners && lower_tournament_winners.length > 0)  {
      lower_tournament_winners.sort(seedCompare); // seed ascending
      // we want upper on front so if there are byes we take them from the top (upper)
      tournament_teams = upper_tournament_losers.concat(lower_tournament_winners);
      lowHigh = true;
    }

    let brackets = makeLowerBracket(tournament_teams, lowHigh);
    for (const bracket of brackets) {
      const team_ids = bracket.teams.map(t => t.team_id);
      logger.info(`bracket round ${round} team_ids ${team_ids}`);
      const seeds = bracket.teams.map(t => t.seed);
      const tournament_seed = Math.min(...seeds);
      await PharahClient.post("/match", 
        { tournament_id,
          tournament_round: round,
          tournament_bracket: "lower",
          tournament_seed: tournament_seed,
          state: bracket.state,
          team_ids: team_ids,
          game_cnt});
      this.track("tournament.create.match", {tournament_id, round: round, state: bracket.state, team_ids: JSON.stringify(team_ids), bracket: "lower", game_cnt});
    }
  }

  async progressTournamentRounds () {
    const startedTournaments = await this.getStartedTournaments();
    for(const tournament of startedTournaments){
      let {league_id, tournament_id, round, round_cnt} = tournament;

      round = parseInt(round);
      round_cnt = parseInt(round_cnt);

      const matchRes = await PharahClient.get('/tournament/bracket', {tournament_id, tournament_round:round, count: 100});
      if (matchRes.total_results === 0) {
        //no matches lets create them
        logger.info("no matches for current round, creating them");
        if(round === 0) {
          continue;
        }

        if (round === 1) {
          const tournament_teams = await this.getTournamentTeams(tournament_id);
          await this.createFirstUpperBracket(tournament_teams, tournament_id, round, UPPER_GAME_CNT);
        } else {
          const upperWinnerTeamIds = [];
          const upperLoserTeamIds = [];
          const lowerWinnerTeamIds = [];
          const lastMatches = await PharahClient.get('/tournament/match', {tournament_id, tournament_round:round-1, count: 100});
          const lastUpperMatches = 
            lastMatches.result.filter(m => m.tournament_bracket === "upper");
          const lastLowerMatches = 
            lastMatches.result.filter(m => m.tournament_bracket === "lower");


          for (const match of lastUpperMatches) {

            const winnerSet = match.sets.filter(s => s.outcome === "win" || s.outcome === "bye")[0] || null;
            const loserSet = match.sets.filter(s => s.outcome !== "win" && s.outcome !== "bye")[0] || null;
            logger.info("winnerSet", winnerSet);
            if (winnerSet) {
              upperWinnerTeamIds.push(winnerSet.team_id);
              if (loserSet) {
                upperLoserTeamIds.push(loserSet.team_id);
              }
            } else {
              // TODO draw should be higher seed team
              logger.error("progressed round but a match didn't have a winner or bye, might be a draw, picking first?? lol");
              // const randomSet = match.sets.splice(Math.floor(Math.random() * match.sets.length), 1);
              // FIXME
              upperWinnerTeamIds.push(match.sets[0].team_id);
              // upperWinnerTeamIds.push(randomSet.team_id);
              // let otherSet = match.sets[0];
              let otherSet = match.sets[1];
              if (otherSet) {
                upperLoserTeamIds.push(otherSet.team_id);
              }
            }
          }

          for (const match of lastLowerMatches) {

            const winnerSet = match.sets.filter(s => s.outcome === "win" || s.outcome === "bye")[0] || null;
            if (winnerSet) {
              lowerWinnerTeamIds.push(winnerSet.team_id);
            } else {
              // TODO draw should be higher seed team
              logger.error("progressed round but a match didn't have a winner or bye, might be a draw, picking first?? lol");
              // const randomSet = match.sets.splice(Math.floor(Math.random() * match.sets.length), 1);
              lowerWinnerTeamIds.push(match.sets[0].team_id);
              // lowerWinnerTeamIds.push(randomSet.team_id);
            }
          }
          
          let is_final = false;
          let tournament_teams = await this.getTournamentTeams(tournament_id);

          if (lastUpperMatches.length === 1 
            && lastUpperMatches[0].sets.length === 1 
            && lastLowerMatches.length === 1) {

            is_final = true;

            // go to final
            
            let upper_tt_winners = tournament_teams.filter(
              (t) => upperWinnerTeamIds.indexOf(t.team_id) > -1 ||
                    lowerWinnerTeamIds.indexOf(t.team_id) > -1);
            await this.createUpperBracket(upper_tt_winners, tournament_id, round, FINAL_GAME_CNT);
          } else {

            let tournament_teams = await this.getTournamentTeams(tournament_id);
            let upper_tt_winners = tournament_teams.filter(t => upperWinnerTeamIds.indexOf(t.team_id) > -1);
            await this.createUpperBracket(upper_tt_winners, tournament_id, round, UPPER_GAME_CNT);
            
            if (tournament.type === TournamentType.DOUBLE_ELIM) {
              let upper_tt_losers  = tournament_teams.filter(t => upperLoserTeamIds.indexOf(t.team_id) > -1);
              let lower_tt_winners = tournament_teams.filter(t => lowerWinnerTeamIds.indexOf(t.team_id) > -1);
              await this.createLowerBracket(upper_tt_losers, lower_tt_winners, tournament_id, round, LOWER_GAME_CNT);
            }

          }
          try {
            const leagueRes = await PharahClient.get('/league', {league_id});
            const league = leagueRes.result[0];
            const round_notification =  {
              title: is_final? 'Final starting now!': `Round ${round} starting now!`,
              message: `${tournament.name} next round starting`,
              icon_url: league.image_url,
              require_interaction: true,
              tournament_id
            };
            const users = _.flatten(tournament_teams.map(t => t.users));
            const user_ids = users.map(u => u.user_id);
            const campaign_name = 'tournament_next_round';
            PharahClient.post('/user/notification', { notification: round_notification, user_ids, campaign_name });
          } catch (err) {
            logger.error('ERROR SENDING ROUND NOTFICATION', err, err.stack);
          }

        }

      } else {
        //lets see if we need to end the round
        const endedMatches = matchRes.result.filter(m => m.state === "ended");
        if(endedMatches.length === matchRes.result.length){
          logger.info("progress the round");

          if(round >= round_cnt){
            //end the tournament
            logger.info("*** END THE TOURNAMENT ***");
            await PharahClient.put("/tournament", {tournament_id, state: "ended", end_dttm: "$NOW"});
            this.track("tournament.end", {tournament_id, state: "ended"});
            // update medals on winners
            await PharahClient.post("/team/maint", {});

          }else {
            logger.info("*** TOURNAMENT PROGRESS TO NEXT ROUND ***");
            await PharahClient.put("/tournament", {tournament_id, round: round + 1});
            this.track("tournament.progress.round", {tournament_id, round: round + 1});
          }
        }
      }
    }
  }

  async onInterval() {
    let lock = null;
    if (this.running) {
      // don't re-enter loop if we are still running
      return;
    }
    this.running = true;
    try {
      lock = await intervalLock.lock('bracket_tournament_starter', LOCK_TTL);
      await this.setTournamentsToStarted();
      await this.progressTournamentRounds();
      await this.setTournamentsToEnd();
      await this.removeEndedTournamentsFromLeague();
    } catch (error) {
      if (error && error.name === "LockError") {
        logger.debug('Failed to get lock storm_tournament_starter');
      } else {
        logger.error("Interval Error", error, error.stack);
        analytics.writeEvent({
          routing_key: 'tinder.tournament.error',
          error: `${error}`
        });
      }
    } finally {
      if (lock) {
        await lock.unlock();
      }
      this.running = false;
    }
  }

}

const singleton = new TournamentWorkerBracket();
module.exports = singleton;
