const Redlock = require('redlock');
const moment = require('moment');
const TournamentWorkerBase = require('./tournament_worker_base');
const TournamentType = TournamentWorkerBase.TournamentType;
const logger = require('../api/logger');
const PharahClient = require('./pharah_client');
const analytics = require('./analytics.js');
const config = require('../config.js');
const cache = require('bebo-node-commons').RedisCache;
const redisClient = cache.getWriteClient(config.TINDER_REDIS_DB);
const Broadcast = require('../api/broadcast');
const Sentry = require('@sentry/node');

const intervalLock =  new Redlock(
  [redisClient],
  {
    driftFactor: 0.01,
    retryCount: 0,
    retryDelay: 500,
    retryJitter: 200
  }
);

const retryLock =  new Redlock(
  [redisClient],
  {
    driftFactor: 0.01,
    retryCount: 10,
    retryDelay: 500,
    retryJitter: 200
  }
);

const LOCK_TTL = 60 * 1000;
const GAME_CNT = -1;

const BracketType = {
  ALIVE: "alive",
  STORM: "storm",
  ELIMINATED: "eliminated",
  WIN: "win"
};

const makeBracket = (ranks) => {
  // requires rank sorted list
  let bracket = [];
  let length = ranks.length;
  if(length === 0){
    return [];
  }

  while(ranks.length > 0) {
    let a = ranks.shift();
    let match = {
      state: "created",
      teams: [a]
    };
    bracket.push(match);
  }
  return bracket;
};

const rankCompare = (a, b) => {
  return b.rank - a.rank;
};

const RELEVANT_URLS = [
  '/tournament/storm/move',
];


class TournamentWorkerStorm extends TournamentWorkerBase {

  constructor() {
    super();
    this.subscribeUrls = RELEVANT_URLS;
    this.onInterval = this.onInterval.bind(this);
    this.onMessage = this.onMessage.bind(this);
    this.supportedTypes = [TournamentType.STORM];
  }

  calculateRounds(tournament) {
    let rounds = tournament.config.rounds;
    return rounds && rounds.length || 1;
  }

  calculateFirstRound(tournament, teamCount) {
    let rounds = tournament.config.rounds;
    let doneRounds = rounds.filter((r) => r.survivors >= teamCount);
    let first = Math.min(doneRounds.length, rounds.length - 1) + 1;
    return first;
  }

  calculateNextRoundDttm(tournament) {

    let nextRoundDttm = null;
    let currentStorm = tournament.config.rounds[tournament.round - 1] || {};
    let lastRoundDttm = tournament.start_dttm;
    if (tournament.next_round_dttm) {
      lastRoundDttm = tournament.next_round_dttm;
    }
    if(currentStorm.seconds) {
      nextRoundDttm = new moment(lastRoundDttm).add(currentStorm.seconds, 'seconds');
    }
    return nextRoundDttm;
  }

  async progressTournamentRounds() {
    const startedTournaments = await this.getStartedTournaments();
    for(const tournament of startedTournaments){
      let {tournament_id, round, round_cnt} = tournament;
      let currentStorm = tournament.config.rounds[tournament.round - 1] || {};
      logger.debug(tournament_id, "progressTournamentRounds", round, round_cnt, currentStorm);

      round = parseInt(round);
      round_cnt = parseInt(round_cnt);

      const matchRes = await PharahClient.get('/tournament/storm', {tournament_id, tournament_round:round, count: 2000});
      if (matchRes.total_results === 0) {
        //no matches lets create them
        logger.info(tournament_id, "no matches for current round, creating them", round, round_cnt);
        if (round === 0) {
          continue;
        }

        const lastMatches = await PharahClient.get('/tournament/storm', {tournament_id, tournament_round:round-1, count: 2000});
        if (round === 1 || lastMatches.result.length === 0) {
          const tournament_teams = await this.getTournamentTeams(tournament_id);
          await this.createFirstStorm(tournament_teams, tournament_id, round, GAME_CNT, currentStorm.survivors);
        } 

      } else {
        // we have matches...
        let cutoff = new moment(tournament.next_round_dttm);
        let now = moment();

        const endedMatches = matchRes.result.filter(m => m.state === "ended");
        if (endedMatches.length !== matchRes.result.length && cutoff.isBefore(now)) {
          logger.info(tournament_id, "*** CUTOFF MET, END ALL MATCHES ***", round, round_cnt);
          for (let m of matchRes.result) {
            if (m.state === "ended") {
              continue;
            }
            let user_id = m.sets[0].team.users[0].user_id; // hate this but match worker needs this to idenfify set
            let ended_dttm = new Date().toISOString();
            await PharahClient.post('/match/forfeit', {match_id: m.match_id, ended_dttm, user_id});
          }

        } else if (endedMatches.length === matchRes.result.length){

          logger.info(tournament_id, "all matches done - *** TOURNAMENT PROGRESS TO NEXT ROUND ***", round + 1);
          if (round >= round_cnt){
            //end the tournament
            logger.info(tournament_id, "*** END THE TOURNAMENT ***", round, round_cnt);
            await this.pickWinner(matchRes.result, round, tournament_id);
            await PharahClient.put("/tournament", {tournament_id, state: "ended", end_dttm: "$NOW"});
            this.track("tournament.end", {tournament_id, state: "ended"});
            // update medals on winners
            await PharahClient.post("/team/maint", {});
          } else {
            let modified = await this.endStorm(matchRes.result, round, tournament_id);
            if (modified) {
              logger.info(tournament_id, "ended storm - *** TOURNAMENT PROGRESS TO NEXT ROUND ***", round + 1);
              continue;
            } else {
              logger.info(tournament_id, "*** PROGRESS THE ROUND ***", round, "->", round+1, round_cnt);
              await this.createNextStorm(matchRes.result, tournament_id, round+1, GAME_CNT, currentStorm.survivors);
              let nextRoundDttm = this.calculateNextRoundDttm(tournament);
              await PharahClient.put("/tournament", 
                { tournament_id,
                  round: round + 1,
                  next_round_dttm: nextRoundDttm});
              this.track("tournament.progress.round",
                { tournament_id,
                  round: round + 1,
                  next_round_dttm: nextRoundDttm});
            }
          }
        }
      }
    }
  }

  async createFirstStorm(tournament_teams, tournament_id, round, game_cnt, survivors) {
    logger.info(tournament_id, "createFirstStorm", round);

    tournament_teams.sort(rankCompare);
    tournament_teams.map(async (team, i) => {
      team.seed = i+1;
      await PharahClient.put("/tournament/team", {tournament_id, team_id: team.team_id, seed: team.seed});
      this.track("tournament.update.team", {tournament_id, team_id: team.team_id, seed: team.seed});
    });

    logger.info("tournament_teams", tournament_teams);

    let brackets = makeBracket(tournament_teams);
    let length = brackets.length;
    let payload = {
      url: "/tournament/storm",
      tournament_id,
    };
    let p = [];
    for (let i=0; i<length; i++) {
      let bracket = brackets[i];
      const team_ids = bracket.teams.map(t => t.team_id);
      const seeds = bracket.teams.map(t => t.seed);
      const tournament_seed = Math.min(...seeds);
      let tournament_bracket = BracketType.ALIVE;
      if (i >= survivors) {
        tournament_bracket = BracketType.STORM;
      }
      logger.info(`bracket round ${round} team_ids ${team_ids}`);
      let r = PharahClient.post("/match", 
        { tournament_id,
          tournament_round: round,
          tournament_seed,
          tournament_bracket,
          tournament_type: TournamentType.STORM,
          state: bracket.state,
          team_ids: team_ids,
          game_cnt});
      p.push(r);
      this.track("tournament.create.match", {tournament_id, round: round, state: bracket.state, team_ids: JSON.stringify(team_ids), bracket: "upper", game_cnt});
    }
    p = await Promise.all(p);
    payload.result = p.map(r => r.result[0]);
    Broadcast.emit("/tournament/storm", tournament_id, payload);
  }

  async createNextStorm(lastMatches, tournament_id, round, game_cnt, survivors) {
    logger.info(tournament_id, "createNextStorm", round);

    let payload = {
      url: "/tournament/storm",
      tournament_id,
    };
    let p = [];

    // should already be in the correct order
    let length = lastMatches.length;
    for (let i=0; i<length; i++) {
      // logger.info(`bracket round ${round} team_ids ${team_ids}`);
      let oldMatch = lastMatches[i];
      let team_id = oldMatch.sets[0].team_id;
      let state = "eliminated";
      let tournament_bracket = BracketType.ELIMINATED;
      if (oldMatch.tournament_bracket === BracketType.ALIVE) {
        state = "created";
        if (i < survivors) {
          tournament_bracket = BracketType.ALIVE;
        } else {
          tournament_bracket = BracketType.STORM;
        }
      }
      let r = PharahClient.post("/match", 
        { tournament_id,
          tournament_round: round,
          tournament_bracket,
          tournament_type: TournamentType.STORM,
          tournament_seed: i,
          state,
          team_ids: [team_id],
          GAME_CNT});
      p.push(r);
      this.track("tournament.create.match", {tournament_id, round: round, state: state, team_ids: team_id});
    } 
    p = await Promise.all(p);
    payload.result = p.map(r => r.result[0]);
    Broadcast.emit("/tournament/storm", tournament_id, payload);
  }

  async onMoveStorm(msg, tournament) {

    // TODO send event
    if (!msg.result) {
      logger.error("no result?", tournament, msg);
      return;
    }
    let round = msg.result[0].round;
    let match_id = msg.result[0].match_id;

    if (round == 0) {
      logger.error("unexpected move on round 0", tournament, msg);
      return;
    }
    const tournament_id = tournament.tournament_id;
    if (!tournament.config.rounds) {
      logger.error(tournament_id, "tournament is missing config.rounds", tournament);
      return;
    }
    if (round > tournament.config.rounds.length) {
      logger.error(tournament_id, "tournament is missing config.rounds for round", round, tournament);
      return;
    }
    let currentStorm = tournament.config.rounds[round - 1];
    if (! currentStorm) {
      logger.error("CURRENT STORM CONFIG INVALID", currentStorm, round, msg, tournament);
      return;
    }
    // TOD  we should only get alive & storm ?
    const res = await PharahClient.get('/tournament/storm',
      { tournament_id, tournament_round: round, count:2000 });
    const matches = res.result;
    let payload = {
      url: "/tournament/storm",
      tournament_id,
    };
    let p = [];

    let length = matches.length;
    for (let i=0; i<length; i++) {
      let m = matches[i];
      if (i < currentStorm.survivors) {
        if (m.tournament_bracket === BracketType.STORM) {
          m.tournament_bracket = BracketType.ALIVE;
          let matchUpdate = {
            match_id: m.match_id,
            tournament_bracket: BracketType.ALIVE,
          };
          let r = PharahClient.put('/match', matchUpdate);
          p.push(r);
          if (match_id === m.match_id) {
            match_id = null;
          }
        }
      } else {
        if (m.tournament_bracket === BracketType.ALIVE) {
          m.tournament_bracket = BracketType.STORM;
          let matchUpdate = {
            match_id: m.match_id,
            tournament_bracket: BracketType.STORM,
          };
          let r = PharahClient.put('/match', matchUpdate);
          p.push(r);
          if (match_id === m.match_id) {
            match_id = null;
          }
        }
      }
    }
    if (match_id) {
      let r = PharahClient.get('/match', {match_id});
      p.push(r);
    }
    p = await Promise.all(p);
    payload.result = p.map(r => r.result[0]);
    Broadcast.emit("/tournament/storm", tournament_id, payload);

  }

  async endStorm(matches, round, tournament_id) {
    let modified = false;

    let payload = {
      url: "/tournament/storm",
      tournament_id,
    };
    let p = [];

    for (let m of matches) {

      let outcome = BracketType.ELIMINATED;
      if (m.tournament_bracket === BracketType.ALIVE) {
        outcome = BracketType.ALIVE;
      }

      for (const set of m.sets) {
        if (set.outcome === outcome) {
          continue;
        }
        modified = true;
        await PharahClient.put('/set', {
          set_id: set.set_id,
          outcome
        });
      }
      if (m.tournament_bracket === outcome) {
        continue;
      }
      modified = true;
      let matchUpdate = {
        match_id: m.match_id,
        tournament_bracket: outcome
      };
      let r = PharahClient.put('/match', matchUpdate);
      p.push(r);
    }
    if (modified) {
      p = await Promise.all(p);
      payload.result = p.map(r => r.result[0]);
      Broadcast.emit("/tournament/storm", tournament_id, payload);
    }
    return modified;
  }

  async pickWinner(matches, round, tournament_id) {
    logger.info(tournament_id, "pickWinner", round);
    let payload = {
      url: "/tournament/storm",
      tournament_id,
    };
    let p = [];

    for (let i=0; i < matches.length; i++) {
      const m = matches[i];
      if (m.tournament_bracket === BracketType.ELIMINATED) {
        continue;
      }
      let outcome = BracketType.ELIMINATED;
      let bracket = BracketType.ELIMINATED;
      if (i === 0) {
        outcome = "win";
        bracket = BracketType.WIN;
      }

      for (const set of m.sets) {
        await PharahClient.put('/set', {
          set_id: set.set_id,
          outcome
        });
      }
      let matchUpdate = {
        match_id: m.match_id,
        tournament_bracket: bracket
      };
      let r = PharahClient.put('/match', matchUpdate);
      p.push(r);
    }
    p = await Promise.all(p);
    payload.result = p.map(r => r.result[0]);
    Broadcast.emit("/tournament/storm", tournament_id, payload);
  }

  async onInterval() {
    let lock = null;
    if (this.running) {
      // don't re-enter loop if we are still running
      return;
    }
    this.running = true;
    try {
      lock = await intervalLock.lock('storm_tournament_starter', LOCK_TTL);
      await this.setTournamentsToStarted();
      await this.progressTournamentRounds();
      await this.setTournamentsToEnd();
      await this.removeEndedTournamentsFromLeague();
    } catch (error) {
      if (error && error.name === "LockError") {
        logger.debug('Failed to get lock storm_tournament_starter');
      } else {
        logger.error("Interval Error", error, error.stack);
        analytics.writeEvent({
          routing_key: 'tinder.tournament.error',
          error: `${error}`
        });
      }
    } finally {
      if (lock) {
        await lock.unlock();
      }
      this.running = false;
    }
  }

  async onMessage(msg) {
    let lock = null;
    let tournament;
    let startTime = process.hrtime();
    let tournament_id = null;
    let lockTime = null;
    let lock_ms = 0;

    try {
      tournament_id = msg.tournament_id;
      if (!tournament_id){
        return;
        // throw new Error(`no tournament_id ${msg}`);
      }
      let fn = null;
      if (msg.url === '/tournament/storm/move') {
        fn = this.onMoveStorm;
      }
      if (fn == null) {
        return;
      }
      // when there is a match update we re-calculate the 
      let startLock = process.hrtime();
      lock = await retryLock.lock(`tournament_${tournament_id}`, LOCK_TTL);
      lockTime = process.hrtime(startLock);

      const res = await PharahClient.get('/tournament', { tournament_id });
      tournament = res.result[0];
      // logger.info("onMessage tournament", tournament, msg);
      await fn(msg, tournament);

    } catch (error) {
      if (error.name === 'LockError') {
        logger.error('Failed to get lock', msg.url, tournament_id);
      } else {
        logger.error('Message processing error', msg, tournament, error);
        Sentry.captureException(error);
      }
      analytics.writeEvent({
        routing_key: 'tinder.tournament.error',
        error: `${error}`,
        tournament
      });
    } finally {
      if(lock){
        await lock.unlock();
      }
      let diff = process.hrtime(startTime);
      let elapsed_ms = (diff[0] * 1e9 + diff[1]) / 1e6;
      if (lockTime) {
        lock_ms = (lockTime[0] * 1e9 + lockTime[1]) / 1e6;
      }
      logger.info('>>>', msg.url, tournament_id, elapsed_ms, "/", elapsed_ms - lock_ms, 'ms');
    }
  }

}

const singleton = new TournamentWorkerStorm();
module.exports = singleton;
