#
# Module for adding external dependencies to a project
#
# VARIABLES
#
#   EXTERNAL_DEPENDENCIES - List of all of the external projects added by
#     add_external_dependency
#
#   EXTERNAL_DEPENDENCY_DIR - Path where all of the external dependencies
#     are downloaded and built
#
#   EXTERNAL_DEPENDENCY_COMMON_CMAKE_OPTIONS - Common CMake options that should
#     be passed from the project's build to the external dependencies
#
# FUNCTIONS
#
#   add_external_dependency(<project> ...) - Add an external dependency to
#     integrate with the current project.  The unparsed arguments are forwarded
#     directly to `ExternalProject_Add()`
#
#   add_external_library(<library> <installed_lib_dir>) - Adds a static, imported
#     library from an external project
#

include(ExternalProject)
include(VerboseMessage)

# Don't overwrite EXTERNAL_DEPENDENCIES since this module can be included
# multiple times
if(NOT DEFINED EXTERNAL_DEPENDENCIES)
  set(EXTERNAL_DEPENDENCIES "" CACHE INTERNAL "External Dependency Projects")
endif()
set(EXTERNAL_DEPENDENCY_DIR ${CMAKE_BINARY_DIR}/external)

set(EXTERNAL_DEPENDENCY_COMMON_CMAKE_OPTIONS
  CMAKE_BUILD_TYPE
  CMAKE_C_COMPILER
  CMAKE_C_FLAGS
  CMAKE_CXX_COMPILER
  CMAKE_CXX_FLAGS
  CMAKE_TOOLCHAIN_FILE
)

if(TARGET_PLATFORM STREQUAL "Android")
  list(APPEND EXTERNAL_DEPENDENCY_COMMON_CMAKE_OPTIONS
    ANDROID_ABI
    ANDROID_NATIVE_API_LEVEL
    ANDROID_STL
    ANDROID_STL_FORCE_FEATURES
    ANDROID_TOOLCHAIN_NAME
  )
endif()


if(ENABLE_BITCODE)
  set(CMAKE_C_FLAGS_RELEASE "${CMAKE_C_FLAGS_RELEASE} -fembed-bitcode")
  set(CMAKE_CXX_FLAGS_RELEASE "${CMAKE_CXX_FLAGS_RELEASE} -fembed-bitcode")
  list(APPEND EXTERNAL_DEPENDENCY_COMMON_CMAKE_OPTIONS
    CMAKE_C_FLAGS_RELEASE
    CMAKE_CXX_FLAGS_RELEASE
  )
endif()

foreach(option ${EXTERNAL_DEPENDENCY_COMMON_CMAKE_OPTIONS})
  if(DEFINED ${option} AND NOT ${option} STREQUAL "")
    list(APPEND EXTERNAL_DEPENDENCY_COMMON_CMAKE_ARGS -D${option}=${${option}})
  endif()
endforeach()

list(APPEND EXTERNAL_DEPENDENCY_COMMON_CMAKE_ARGS -DCMAKE_INSTALL_PREFIX=${EXTERNAL_DEPENDENCY_DIR})

function(add_external_dependency project)
  message(STATUS "Adding external dependency: ${project}")

  set(options "")
  set(oneValueArgs "")
  set(multiValueArgs "")
  cmake_parse_arguments(ADD_EXT_DEP_ARGS "${options}" "${oneValueArgs}" "${multiValueArgs}" ${ARGN})

  set_directory_properties(PROPERTIES EP_PREFIX ${EXTERNAL_DEPENDENCY_DIR})

  ExternalProject_Add(${project}
    CMAKE_COMMAND ${CMAKE_COMMAND}
    CMAKE_GENERATOR "${CMAKE_GENERATOR}"
    ${ADD_EXT_DEP_ARGS_UNPARSED_ARGUMENTS}
    UPDATE_DISCONNECTED 1
    TIMEOUT 60
  )

  # Reset EP_PREFIX for other external projects
  set_directory_properties(PROPERTIES EP_PREFIX "")

  # Add dependency to global list
  set(dependencies ${EXTERNAL_DEPENDENCIES})
  list(APPEND dependencies ${project})
  list(REMOVE_DUPLICATES dependencies)
  set(EXTERNAL_DEPENDENCIES ${dependencies} CACHE INTERNAL "External Dependency Projects")
endfunction()
