#
# CMake module to help add code coverage flags and targets
#
#
# OUTPUTS:
#
# Variables(s):
#
#   ${CMAKE_PROJECT_NAME}_OBJECT_LIBRARIES - Global list of all of the object libraries
#
#
# Function(s):
#
#   add_submodule(<library_name>
#     SOURCES [<files>...]
#     LINK_LIBRARIES [<libraries>...]
#   )
#
#     The goal of this function is to mask the creation of an object library while
#     still creating the normal library.  THe parameters are defined to be:
#
#       SOURCES: The source files of the library
#       LINK_LIBRARIES: THe libraries that the library will link against
#
#     Here is an example of how the function should be invoked:
#
#       add_submodule(library SOURCES src1 src2 LINK_LIBRARIES lib1 lib2)
#
#     Which should be functionally similar to the following CMake syntax:
#
#       add_library(library src1 src2)
#       add_dependencies(library dependency)
#       target_link_libraries(library lib1 lib2)
#
#
#   create_unified_library(<library_name> [SHARED]
#     [SOURCES <file1> <file2>...]
#   )
#
#     Creates a unified library from all of the ${CMAKE_PROJECT_NAME}_OBJECT_LIBRARIES
#

include(TargetFolderProperty)
include(VerboseMessage)

set(OBJECT_LIBRARIES_NAME "${CMAKE_PROJECT_NAME}_OBJECT_LIBRARIES")
set(OBJECT_LIBRARIES_DESCRIPTION "Object libraries for ${CMAKE_PROJECT_NAME}")
set(${OBJECT_LIBRARIES_NAME} "" CACHE INTERNAL ${OBJECT_LIBRARIES_DESCRIPTION})
set(PLAYERCORE_LIBRARIES "" CACHE INTERNAL "Player Core libraries/submodules")

function(add_submodule library_name)
  cmake_parse_arguments(
    PARSED_ARGS
    ""
    ""
    "SOURCES;LINK_LIBRARIES"
    ${ARGN}
  )

  if(NOT DEFINED PARSED_ARGS_SOURCES)
    message(FATAL_ERROR "Must specify at least one source file (SOURCES)")
  endif()

  file(RELATIVE_PATH relative_dir ${CMAKE_SOURCE_DIR} ${CMAKE_CURRENT_SOURCE_DIR})

  if(BUILD_UNIFIED_LIB)
    set(object_library ${library_name}_obj)

    verbose_message("Adding object library: ${object_library}")
    add_library(${object_library} OBJECT ${PARSED_ARGS_SOURCES})
    set(${OBJECT_LIBRARIES_NAME} ${${OBJECT_LIBRARIES_NAME}} ${object_library} CACHE INTERNAL ${OBJECT_LIBRARIES_DESCRIPTION})
    set_folder_property(${object_library})

    foreach(dep ${PARSED_ARGS_LINK_LIBRARIES})
      if(TARGET ${dep})
        get_target_property(imported ${dep} IMPORTED)
        if(${imported})
          add_dependencies(${object_library} ${dep})
        endif()
        target_include_directories(${object_library} PRIVATE $<TARGET_PROPERTY:${dep},INCLUDE_DIRECTORIES>)
      endif()
    endforeach(dep)

    add_library(${library_name} $<TARGET_OBJECTS:${object_library}>)
    set_target_properties(${library_name} PROPERTIES LINKER_LANGUAGE CXX)
    set_folder_property(${library_name})

    if(DEFINED PARSED_ARGS_LINK_LIBRARIES)
      target_link_libraries(${library_name} ${PARSED_ARGS_LINK_LIBRARIES})
    endif()
  else()
    add_library(${library_name} ${PARSED_ARGS_SOURCES})

    if(DEFINED PARSED_ARGS_LINK_LIBRARIES)
      target_link_libraries(${library_name} ${PARSED_ARGS_LINK_LIBRARIES})
    endif()
    # Check for imported libraries that need to be explicitly added as dependencies
    foreach(library ${PARSED_ARGS_LINK_LIBRARIES})
      if(TARGET ${library})
        add_dependencies(${library_name} ${library})
      endif()
    endforeach()

    set_folder_property(${library_name})
  endif()

  set(libraries ${PLAYERCORE_LIBRARIES})
  list(APPEND libraries ${library_name})
  set(PLAYERCORE_LIBRARIES ${libraries} CACHE INTERNAL "Player Core libraries/submodules")
endfunction()

function(create_unified_library library_name)
  verbose_message("Creating unified library: ${library_name}")
  cmake_parse_arguments(
    PARSED_ARGS
    "SHARED"
    ""
    "SOURCES"
    ${ARGN}
  )

  set(library_type STATIC)
  if(${PARSED_ARGS_SHARED})
    set(library_type SHARED)
  endif()

  set(library_objects "")
  foreach(object_library ${${OBJECT_LIBRARIES_NAME}})
    list(APPEND library_objects $<TARGET_OBJECTS:${object_library}>)
  endforeach()

  add_library(${library_name} ${library_type} ${PARSED_ARGS_SOURCES} ${library_objects})
endfunction()
