package tv.twitch.player.android.example;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.pm.ActivityInfo;
import android.graphics.Typeface;
import android.media.AudioManager;
import android.net.Uri;
import android.os.Bundle;
import android.text.TextUtils;
import android.util.Log;
import android.view.MotionEvent;
import android.view.Surface;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.AutoCompleteTextView;
import android.widget.Button;
import android.widget.MediaController;
import android.widget.ProgressBar;
import android.widget.Spinner;
import android.widget.TextView;
import android.widget.Toast;

import org.json.JSONException;
import org.json.JSONObject;

import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import tv.twitch.android.player.MediaException;
import tv.twitch.android.player.MediaPlayer;
import tv.twitch.android.player.MediaType;
import tv.twitch.android.player.Player;
import tv.twitch.android.player.Quality;
import tv.twitch.android.player.Statistics;
import tv.twitch.android.player.ViewUtil;

public class PlayerActivity extends Activity implements
        SurfaceHolder.Callback,
        MediaController.MediaPlayerControl,
        Player.Listener {

    private static final String TAG = "Twitch";
    private static final String DEFAULT_STREAM = "https://www.twitch.tv/bufferbunny";

    private SurfaceView surfaceView;
    private SurfaceView previousSurfaceView;
    private Player player;
    private MediaController controller;
    private TextView streamTextView;
    private TextView debugTextView;
    private TextView captionsView;
    private ProgressBar progressView;
    private Spinner spinner;
    private String currentStream;
    private List<String> streams;
    private Surface surface;
    private boolean muted;
    private Runnable statisticsUpdate;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE);
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN);
        setContentView(R.layout.activity_player);

        player = new MediaPlayer(this);
        streams = new ArrayList<>();
        surfaceView = (SurfaceView) findViewById(R.id.surfaceView);
        surfaceView.getHolder().addCallback(this);
        streamTextView = (TextView) findViewById(R.id.textStream);
        debugTextView = (TextView) findViewById(R.id.textVersion);
        captionsView = (TextView) findViewById(R.id.captions);
        captionsView.setTypeface(Typeface.MONOSPACE);
        progressView = (ProgressBar) findViewById(R.id.progress);
        spinner = (Spinner) findViewById(R.id.spinner);
        Spinner rateSpinner = (Spinner) findViewById(R.id.rate_spinner);
        List<Float> rates = Arrays.asList(1.0f, 1.25f, 1.5f, 2.0f);
        final ArrayAdapter<Float> adapter = new ArrayAdapter<>(this,
                android.R.layout.simple_spinner_item, rates);
        adapter.setDropDownViewResource(android.R.layout.simple_dropdown_item_1line);
        rateSpinner.setAdapter(adapter);
        rateSpinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                Float rate = adapter.getItem(position);
                if (player != null) {
                    player.setPlaybackRate(rate);
                }
            }
            @Override
            public void onNothingSelected(AdapterView<?> parent) {
            }
        });

        streamTextView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                final AutoCompleteTextView urlEdit = new AutoCompleteTextView(PlayerActivity.this);
                urlEdit.setAdapter(new ArrayAdapter<>(PlayerActivity.this,
                        android.R.layout.simple_dropdown_item_1line, streams));
                new AlertDialog.Builder(PlayerActivity.this)
                        .setView(urlEdit).setPositiveButton("OK",
                        new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        String url = urlEdit.getText().toString();
                        if (!TextUtils.isEmpty(url)) {
                            loadUrl(url);
                        }
                    }
                }).create().show();
            }
        });
        final Button mute = (Button) findViewById(R.id.mute);
        mute.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                player.setVolume(muted? 1.0f : 0.0f);
                mute.setText(muted? "Mute" : "Unmute");
                muted = !muted;
            }
        });
        final Button switchSurface = (Button) findViewById(R.id.switch_surface);
        switchSurface.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                //player.setSurface(null);
                ViewGroup parent = (ViewGroup) surfaceView.getParent();
                previousSurfaceView = surfaceView;
                surfaceView.getHolder().removeCallback(PlayerActivity.this);
                surfaceView = new SurfaceView(PlayerActivity.this);
                surfaceView.getHolder().addCallback(PlayerActivity.this);
                parent.addView(surfaceView);
            }
        });
        controller = new MediaController(PlayerActivity.this, false);
        controller.setAnchorView(surfaceView);
        controller.setMediaPlayer(PlayerActivity.this);

        // focus audio stream
        int streamType = AudioManager.STREAM_MUSIC;
        setVolumeControlStream(streamType);
        AudioManager audioManager = (AudioManager) getSystemService(Context.AUDIO_SERVICE);
        audioManager.requestAudioFocus(null, streamType, AudioManager.AUDIOFOCUS_GAIN);

        statisticsUpdate = new Runnable() {
            @Override
            public void run() {
                if (player != null && player.getState() == Player.State.PLAYING) {
                    updateDebugText();
                    surfaceView.postDelayed(statisticsUpdate, 1000);
                }
            }
        };
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        boolean handled = super.onTouchEvent(event);
        if (controller != null && !handled && !controller.isShowing()) {
            controller.show();
        }

        return handled;
    }

    @Override
    protected void onStart() {
        super.onStart();
        player.addListener(this);
        if (surface != null) {
            player.setSurface(this.surface);
            loadUrl(currentStream);
        }
        if (currentStream == null) {
            currentStream = DEFAULT_STREAM;
        }
    }

    @Override
    protected void onStop() {
        super.onStop();
        if (player != null) {
            player.pause();
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        if (player != null) {
            player.removeListener(this);
            player.release();
        }
    }

    @Override
    public void start() {
        if (player != null) {
            player.play();
        }
    }

    @Override
    public void pause() {
        if (player != null) {
            player.pause();
        }
    }

    @Override
    public int getDuration() {
        return player == null ? 0 : (int) player.getDuration();
    }

    @Override
    public int getCurrentPosition() {
        return player == null ? 0 : (int) player.getPosition();
    }

    @Override
    public void seekTo(int pos) {
        if (player != null && player.getDuration() > 0) {
            player.seekTo(pos);
        }
    }

    @Override
    public boolean isPlaying() {
        return player != null && player.getState() == Player.State.PLAYING;
    }

    @Override
    public int getBufferPercentage() {
        if (isPlaying() && player.getDuration() > 0) {
            long buffered = player.getBufferedPosition();
            float bufferPercent = (buffered / (float)(player.getDuration()));
            return (int) (bufferPercent * 100);
        }
        return 0;
    }

    @Override
    public boolean canPause() {
        return true;
    }

    @Override
    public boolean canSeekBackward() {
        return player != null && player.getDuration() > 0;
    }

    @Override
    public boolean canSeekForward() {
        return player != null && player.getDuration() > 0;
    }

    @Override
    public int getAudioSessionId() {
        return player == null ? 0 : player.getAudioSessionId();
    }

    @Override
    public void onStateChanged(final Player.State state) {
        if (isFinishing()) {
            return;
        }
        switch (state) {
            case BUFFERING:
                progressView.setVisibility(View.VISIBLE);
                break;
            case READY:
                player.play();
                updateSpinner();
                progressView.setVisibility(View.INVISIBLE);
                break;
            case IDLE:
                progressView.setVisibility(View.INVISIBLE);
                break;
            case PLAYING:
                surfaceView.postDelayed(statisticsUpdate, 1000);
                progressView.setVisibility(View.INVISIBLE);
                break;
        }
    }

    @Override
    public void onError(final MediaException exception) {
        String text = exception.getMessage();
        Toast.makeText(PlayerActivity.this, text, Toast.LENGTH_SHORT).show();
    }

    @Override
    public void onMetadata(String type, final ByteBuffer data) {
        if (type.equalsIgnoreCase(MediaType.TEXT_JSON)) {
            String content = Charset.forName("UTF-8").decode(data).toString();
            try {
                JSONObject json = new JSONObject(content);
                if (json.has("caption")) {
                    json = json.getJSONObject("caption");
                    if (json.has("text")) {
                        captionsView.setText(json.getString("text"));
                        return;
                    }
                }
                Log.v(TAG, "Player metadata " + type + " content: " + content);
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public void onQualityChanged(Quality quality) {
        Log.i(TAG, "Quality changed to " + quality.getName());
        updateSpinner();
    }

    @Override
    public void onRebuffering() {
        Log.w(TAG, "Player re-buffering");
    }

    @Override
    public void onAnalyticsEvent(String name, String properties) {
        Log.i(TAG, "Player analytics " + name + " : " + properties);
    }

    @Override
    public void onVideoSizeChanged(final int width, final int height) {
        Log.i(TAG, "video size changed to " + width + " x " + height);
        if (!isFinishing()) {
            ViewUtil.setLayoutParams(surfaceView, width, height);
        }
    }

    private void updateSpinner() {
        String auto = "auto";
        String currentQuality = player.getQuality().getName();
        if (player.getAutoSwitchQuality() && !TextUtils.isEmpty(currentQuality)) {
            auto += " (" + currentQuality + ")";
        }
        int selected = 0;
        ArrayList<String> names = new ArrayList<>();
        for (Quality quality : player.getQualities()) {
            names.add(quality.getName());
        }
        Collections.sort(names);
        names.add(0, auto);
        if (!player.getAutoSwitchQuality()) {
            for (int i = 0; i < names.size(); i++) {
                if (names.get(i).equalsIgnoreCase(currentQuality)) {
                    selected = i;
                }
            }
        }
        final ArrayAdapter<String> adapter = new ArrayAdapter<>(this,
                android.R.layout.simple_spinner_item, names);
        adapter.setDropDownViewResource(android.R.layout.simple_dropdown_item_1line);
        spinner.setOnItemSelectedListener(null);
        spinner.setAdapter(adapter);
        spinner.setSelection(selected, false);
        spinner.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> parent, View view, int position, long id) {
                String name = adapter.getItem(position);

                if (name != null && name.startsWith("auto")) {
                    player.setAutoSwitchQuality(true);
                } else {
                    for (Quality quality : player.getQualities()) {
                        if (quality.getName().equalsIgnoreCase(name)) {
                            player.setQuality(quality);
                            player.setAutoSwitchQuality(false);
                            break;
                        }
                    }
                }
            }
            @Override
            public void onNothingSelected(AdapterView<?> parent) {
            }
        });
    }

    private void loadUrl(String url) {
        if (surface != null && url != null) {
            streamTextView.setText(url);
            if (!streams.contains(url)) {
                streams.add(url);
            }

            player.load(Uri.parse(url));
        }
    }

    private void updateDebugText() {
        String text = "v" + player.getVersion();
        long buffered = (player.getBufferedPosition() - player.getPosition());
        text += "\nBuffer " + buffered + " ms";
        if (player.getDuration() == -1) {
            text += "\nLive Latency " + player.getLiveLatency() + " ms";
        }
        float kbps = (player.getBandwidthEstimate() / 1000.f);
        text += "\nBandwidth estimate " + kbps + " kbps";
        Statistics statistics = player.getStatistics();
        text += "\nFrames dropped " + statistics.getDroppedFrames() + " / " + statistics.getDecodedFrames();
        text += "\nBitrate " + statistics.getVideoBitRate() / 1000.f + " kbps";
        String codecs = player.getQuality().getCodecs();
        if (!TextUtils.isEmpty(codecs)) {
            text += "\nCodecs " + player.getQuality().getCodecs();
        }
        debugTextView.setText(text);
    }

    @Override
    public void surfaceCreated(SurfaceHolder holder) {
        this.surface = holder.getSurface();
        if (player != null) {
            player.setSurface(this.surface);
            if (player.getState() == Player.State.IDLE) {
                loadUrl(currentStream);
            }
            if (previousSurfaceView != null) {
                ViewGroup parent = (ViewGroup) surfaceView.getParent();
                parent.removeView(previousSurfaceView);
                previousSurfaceView = null;
            }
        }
    }

    @Override
    public void surfaceChanged(SurfaceHolder holder, int format, int width, int height) {
    }

    @Override
    public void surfaceDestroyed(SurfaceHolder holder) {
        this.surface = null;
        if (player != null) {
            player.setSurface(null);
        }
    }
}
