#import "ViewController.h"
#import <AVFoundation/AVFoundation.h>
#import <AVKit/AVKit.h>
#import <CoreMedia/CoreMedia.h>
#import <MediaPlayer/MediaPlayer.h>
#import <TTVQuality.h>

@implementation ViewController {
    CADisplayLink* playbackUpdater; // The updater for the time/duration label
    TTVPlayer* _player;
    NSDateComponentsFormatter* timeFormatter;
}

- (BOOL)shouldAutorotate
{
    return YES;
}

- (void)viewWillDisappear:(BOOL)animated
{
    [super viewWillDisappear:animated];
}

- (CALayer*)createLayer
{
    BOOL loopbackMode = NO;
    CALayer* layer = nil;
    if (loopbackMode) {
        layer = (AVPlayerLayer*)[[AVPlayerLayer alloc] init];
    } else {
        layer = (TTVPlayerLayer*)[[TTVPlayerLayer alloc] init];
        //layer = (AVSampleBufferDisplayLayer*)[[AVSampleBufferDisplayLayer alloc] init];
    }
    [self.playerView.layer addSublayer:layer];
    layer.frame = self.playerView.bounds;
    return layer;
}

- (void)initPlayer
{
    _player = [[TTVPlayer alloc] initWithLayer:[self createLayer]];
    _player.delegate = self;

    //Add volume view
#if TARGET_OS_IOS
    self.volumeView.backgroundColor = [UIColor clearColor];
    MPVolumeView *mpVolumeView = [[MPVolumeView alloc] initWithFrame: self.volumeView.bounds];
    [self.volumeView addSubview: mpVolumeView];
#endif

    if ([_player.layer isKindOfClass:[AVPlayerLayer class]]) {
        if ([self respondsToSelector:@selector(setupPictureInPicture:)]) {
            [self performSelector:@selector(setupPictureInPicture:) withObject:(AVPlayerLayer*)_player.layer];
        }
        [_player setClosedCaptionDisplayEnabled:YES];
    }

    ViewController* __weak weakSelf = self;
    [_player addPeriodicTimeObserverForInterval:CMTimeMakeWithSeconds(1.0, NSEC_PER_SEC)
                                         queue:nil
                                    usingBlock:^(CMTime time) {
                                        (void)time;
                                        TTVPlayer* player = weakSelf.player;
                                        // update stats display
                                        NSMutableString* text = [NSMutableString stringWithFormat:@"v%@\n", player.version];
                                        CMTime bufferSize = CMTimeSubtract(player.buffered, player.position);
                                        [text appendFormat:@"Buffer %.2f ms\n", CMTimeGetSeconds(bufferSize) * 1000.f];
                                        if (CMTIME_IS_INDEFINITE(player.duration)) {
                                            [text appendFormat:@"Live Latency %.2f ms\n", CMTimeGetSeconds(player.liveLatency) * 1000.f];
                                        }
                                        [text appendFormat:@"Bandwidth estimate %.2f kbps\n", player.bandwidthEstimate / 1000.f];
                                        [text appendFormat:@"Frames dropped %d / %d\n", player.videoFramesDropped, player.videoFramesDecoded];
                                        [text appendFormat:@"Bitrate %.2f kbps\n", player.videoBitrate / 1000.f];
                                        weakSelf.statsLabel.text = text;
                                        weakSelf.statsLabel.numberOfLines = 0;
#if TARGET_OS_IOS
                                        weakSelf.slider.value = CMTimeGetSeconds(time);
#endif
                                    }];

    [_player load:[NSURL URLWithString:_urlTextField.text]];
}

- (void)viewDidLoad
{
    [super viewDidLoad];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(applicationWillResignActive:) name:UIApplicationWillResignActiveNotification object:nil];
    [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(applicationDidBecomeActive:) name:UIApplicationDidBecomeActiveNotification object:nil];
    // Init UI
    [_playPauseButton setTitle:@"\u25B6" forState:UIControlStateNormal];
    [_playPauseButton setTitle:@"\u25A0" forState:UIControlStateSelected];

    playbackUpdater = [CADisplayLink displayLinkWithTarget:self selector:@selector(updateTimeDurationLabel)];
    [playbackUpdater addToRunLoop:[NSRunLoop currentRunLoop] forMode:NSRunLoopCommonModes];
    [playbackUpdater setPaused:YES];

    [self urlTextField].text = @"https://www.twitch.tv/bufferbunny";
    _activityIndicator.hidden = YES;
    _playPauseButton.selected = NO;

    _positionLabel.font = [UIFont monospacedDigitSystemFontOfSize:12.0 weight:UIFontWeightMedium];
    _durationLabel.font = [UIFont monospacedDigitSystemFontOfSize:12.0 weight:UIFontWeightMedium];
    timeFormatter = [[NSDateComponentsFormatter alloc] init];
    timeFormatter.zeroFormattingBehavior = NSDateComponentsFormatterZeroFormattingBehaviorPad;
    timeFormatter.allowedUnits = NSCalendarUnitHour | NSCalendarUnitMinute | NSCalendarUnitMinute | NSCalendarUnitSecond;
    timeFormatter.unitsStyle = NSDateComponentsFormatterUnitsStylePositional;

    _urlTextField.delegate = self;

    [self initPlayer];
    [self setupCommandCenter];

    if ([self respondsToSelector:@selector(initPickerView)]) {
        [self performSelector:@selector(initPickerView)];
    }
}

- (void)setupCommandCenter
{
    MPRemoteCommandCenter* commandCenter = [MPRemoteCommandCenter sharedCommandCenter];
    commandCenter.playCommand.enabled = YES;
    [commandCenter.playCommand addTargetWithHandler:^MPRemoteCommandHandlerStatus(MPRemoteCommandEvent* event) {
        [_player play];
        return MPRemoteCommandHandlerStatusSuccess;
    }];

    commandCenter.pauseCommand.enabled = YES;
    [commandCenter.pauseCommand addTargetWithHandler:^MPRemoteCommandHandlerStatus(MPRemoteCommandEvent* event) {
        [_player pause];
        return MPRemoteCommandHandlerStatusSuccess;
    }];
}

- (void)viewWillTransitionToSize:(CGSize)size withTransitionCoordinator:(id<UIViewControllerTransitionCoordinator>)coordinator
{
    [coordinator animateAlongsideTransition:^(id<UIViewControllerTransitionCoordinatorContext> context) {
        _player.layer.frame = self.playerView.bounds;
    }
                                 completion:^(id<UIViewControllerTransitionCoordinatorContext> context){

                                 }];

    [super viewWillTransitionToSize:size withTransitionCoordinator:coordinator];
}

- (IBAction)onPlayPauseButton:(id)sender
{
    (void)sender;

    if (_urlTextField.text == nil || [_urlTextField.text length] == 0) {
        NSLog(@"onLoadButton(): Cannot load empty URL");
        return;
    }

    if (_player.state == TTVPlayerStateIdle && _player.path.absoluteString.length == 0) {
        if (_urlTextField.text == nil || [_urlTextField.text length] == 0) {
            NSLog(@"onLoadButton(): Cannot load empty URL");
            return;
        }
        _activityIndicator.hidden = NO;
        [_player load:[NSURL URLWithString:_urlTextField.text]];
    } else {
        if (![_player.path.absoluteString isEqualToString:_urlTextField.text]) {
            [_player load:[NSURL URLWithString:_urlTextField.text]];
        } else {
            if (_player.state == TTVPlayerStatePlaying) {
                [_player pause];
            } else {
                [_player play];
            }
        }
    }
}

- (IBAction)onSlider:(id)sender
{
    (void)sender;
#if TARGET_OS_IOS
    [_player seekTo:CMTimeMakeWithSeconds(_slider.value, 600)];
#endif
}

- (void)onDurationChanged:(TTVPlayer* _Nonnull)sender
{
    (void)sender;

    if (sender.seekable) {
        _durationLabel.hidden = NO;
        _durationLabel.text = [timeFormatter stringFromTimeInterval:CMTimeGetSeconds(sender.duration)];
    } else {
        _durationLabel.hidden = YES;
    }

#if TARGET_OS_IOS
    _slider.hidden = !sender.seekable;
    if (!_slider.hidden) {
        _slider.maximumValue = CMTimeGetSeconds(sender.duration);
    }
#endif
}

- (void)onStateChanged:(TTVPlayer* _Nonnull)sender
{
    switch (sender.state) {
    case TTVPlayerStatePlaying: {
        [_activityIndicator setHidden:YES];
        [playbackUpdater setPaused:NO];
        _playPauseButton.selected = YES;
    } break;
    case TTVPlayerStateIdle:
        break;
    case TTVPlayerStateEnded:
        break;
    case TTVPlayerStateReady: {
        [sender play];
        _playPauseButton.enabled = YES;
        _playPauseButton.selected = NO;
        if ([self respondsToSelector:@selector(updateQualityControl)]) {
            [self performSelector:@selector(updateQualityControl)];
        }
    } break;
    case TTVPlayerStateBuffering: {
        [_activityIndicator setHidden:NO];
        _playPauseButton.selected = NO;
    } break;
    }
}

- (void)onAnalyticsEvent:(TTVPlayer*)sender
                        :(NSString*)name
                        :(NSDictionary*)dictionary
{
    (void)sender;
    (void)name;
    (void)dictionary;
}

- (void)onQualityChanged:(TTVPlayer* _Nonnull)sender
                        :(NSString* _Nonnull)name
{
    (void)sender;
    if ([self respondsToSelector:@selector(updatePickerQuality:)]) {
        [self performSelector:@selector(updatePickerQuality:) withObject:(NSString*)name ];
    }
}

- (void)onMetadata:(TTVPlayer* _Nonnull)sender
                  :(NSString* _Nonnull)type
                  :(NSData* _Nonnull)content
{
    (void)sender;
    if ([type caseInsensitiveCompare:@"text/json"] == NSOrderedSame) {
        NSError* jsonError;
        NSDictionary* dict = [NSJSONSerialization JSONObjectWithData:content
                                                             options:NSJSONReadingMutableContainers
                                                               error:&jsonError];
        if (!jsonError && false) {
            NSLog(@"player metadata %@", [dict description]);
        }
    }
}

- (void)onSeekCompleted:(TTVPlayer* _Nonnull)sender
                       :(CMTime)time
{
    (void)sender;
    (void)time;
}

- (void)onError:(TTVPlayer* _Nonnull)sender
               :(NSError* _Nonnull)error
{
    (void)sender;
    (void)error;
}

- (void)onExternalPlaybackChanged:(TTVPlayer* _Nonnull)sender
{
    (void)sender;
}

- (void)onPlayerItemChanged:(TTVPlayer* _Nonnull)sender
                           :(AVPlayerItem*)playerItem
{
    (void)sender;
    (void)playerItem;
}

#pragma mark - Private

- (void)updateTimeDurationLabel
{
    if (_player) {
        _positionLabel.text = [timeFormatter stringFromTimeInterval:CMTimeGetSeconds(_player.position)];
    }
}

- (BOOL)textFieldShouldReturn:(UITextField*)textField
{
    (void)textField;
    [self->_urlTextField resignFirstResponder];
    return YES;
}

- (void)applicationWillResignActive:(NSNotification*)note
{
    [_player pause];
    [_player.layer removeFromSuperlayer];
    _player.layer = nil;
}

- (void)applicationDidBecomeActive:(NSNotification*)note
{
    if (_player.layer == nil) {
        _player.layer = [self createLayer];
    }
    if (_player.state == TTVPlayerStateIdle) {
        [_player play];
    }
}

- (TTVPlayer*)player
{
    return _player;
}

@end
