#include "options.hpp"
#include "optionparser.hpp"
#include <cstdlib>
#include <iostream>
#include <vector>

using namespace twitch;

bool Options::parse(int argc, const char* argv[])
{
    enum optionIndex {
        UNKNOWN,
        HELP,
        URL,
        STOPAFTER,
        DEFAULT_QUALITY,
        SEEKTO,
        DUMP_SEGMENTS,
        USE_FRAMELOGGER,
        DISABLE_ABS,
    };
    const option::Descriptor usage[] = {
        { UNKNOWN, 0, "", "", option::Arg::None, "USAGE: nativeplayer [options] [URL to load (default is http://twitch.tv/monstercat)]\n\n"
                                                 "Options:" },
        { HELP, 0, "", "help", option::Arg::None, "  --help  \tPrint usage and exit." },
        { STOPAFTER, 0, "", "stop-after", option::Arg::Integer, "  --stop-after=<frames>  \tStop after X frames." },
        { SEEKTO, 0, "", "seek-to", option::Arg::Integer, "  --seek-to=<seconds> \tSeek to specific time before play." },
        { DEFAULT_QUALITY, 0, "", "default-quality", option::Arg::None, "  --default-quality \tSet source quality before play." },
        { DUMP_SEGMENTS, 0, "", "dump-segments", option::Arg::None, "  --dump-segments \tWrite segments to the disk as they are being read." },
        { USE_FRAMELOGGER, 0, "", "use-framelogger", option::Arg::None, "  --use-framelogger \tDisplay information about media frames as they are being processed." },
        { DISABLE_ABS, 0, "", "disable-abs", option::Arg::None, "  --disable-abs \tDisable abs (automatic bitrate switching). Abs is on by default unless disabled." },
        { 0, 0, 0, 0, 0, 0 }
    };

    argc -= (argc > 0);
    argv += (argc > 0); // skip program name argv[0] if present

    option::Stats stats(usage, argc, argv);
    std::vector<option::Option> options(stats.options_max);
    std::vector<option::Option> buffer(stats.buffer_max);
    option::Parser parse(usage, argc, argv, &options[0], &buffer[0]);

    if (parse.error()) {
        option::printUsage(std::cout, usage);
        return false;
    }

    if (options[HELP]) {
        option::printUsage(std::cout, usage);
        return false;
    }

    int nonOptionsCount = parse.nonOptionsCount();
    if (nonOptionsCount == 0) {
        m_urls.push_back("https://www.twitch.tv/monstercat"); // 24h Live Music
        m_urls.push_back("https://www.twitch.tv/videos/v205550776"); // Short VOD, reported issues with Seek
        m_urls.push_back("https://www.twitch.tv/videos/93348507"); // Bip! Bop!    (Note: The video is actual off by 1 frame) and in mono
        m_urls.push_back("https://www.twitch.tv/videos/133128389"); // Generic VOD from Dark Souls 3
        m_urls.push_back("https://www.twitch.tv/videos/90812612"); // A/V Sync Vod (Bouncing Ball)
        m_urls.push_back("https://www.twitch.tv/videos/4878428"); // Vod with corrupted audio frames, should play fine and skip the bad audio
        m_urls.push_back("https://www.twitch.tv/videos/258615219"); // 7 Audio channels
        m_urls.push_back("https://twitch.tv/videos/402798647"); // VOD with quality change issue

        // Live channels that aren't always streaming
        // m_urls.push_back("https://www.twitch.tv/toad22484");                // Live Channel

        // m_urls.push_back("https://www.twitch.tv/twitchmedia2"); // Twitch Media test

        // Bad URLs
        // m_urls.push_back("https://www.twitch.tv/videos/161856879");         // this VOD has strange visual frames when you --seek-to=2400
        // m_urls.push_back("https://www.twitch.tv/twitchmedia_rv2_07");       // Twitch Media Stream
    } else {
        for (int i = 0; i < nonOptionsCount; ++i) {
            m_urls.push_back(parse.nonOption(i));
        }
    }

    printf("Going to load URL: %s\n", m_urls[0].c_str());

    for (int i = 0; i < parse.optionsCount(); ++i) {
        option::Option& opt = buffer[i];

        switch (opt.index()) {
        case STOPAFTER:
            m_numFramesRun = std::atoi(opt.arg);
            printf("Option: Auto-stopping after %d frames\n", m_numFramesRun);
            break;

        case SEEKTO:
            m_seekTo = std::atoi(opt.arg);
            printf("Option: Seeking to %d seconds\n", m_seekTo);
            break;

        case DEFAULT_QUALITY:
            m_defaultQuality = true;
            printf("Option: Setting source quality\n");
            break;

        case DUMP_SEGMENTS:
            m_dumpSegments = true;
            printf("Option: Dumping segments on disk\n");
            break;

        case USE_FRAMELOGGER:
            m_useFrameLogger = true;
            printf("Option: Using framelogger\n");
            break;

        case DISABLE_ABS:
            m_disableAbs = true;
            printf("Option: Disabling abs\n");
            break;
        }
    }

    return true;
}
