#include "sampleplayer.hpp"
#include "display.hpp"
#include <cassert>
#include <iostream>

#define PLAYER_RELOAD_STRESS_TEST 0

size_t sceUserMainThreadStackSize = 4 * 1024;
extern const char sceUserMainThreadName[] = "SampleVideodec2Main";
int sceUserMainThreadPriority = SCE_KERNEL_PRIO_FIFO_DEFAULT;

using namespace twitch;
using namespace twitch::ps4;

MediaTime randomDuration(MediaTime lowerBound, MediaTime upperBound)
{
    assert(lowerBound != upperBound);
    auto boundDiffMsec = (upperBound - lowerBound).milliseconds();
    return MediaTime(std::chrono::milliseconds(rand() % boundDiffMsec.count())) + lowerBound;
}

// Example:
// nativeplayer -stop-after 3000 -seek-to 2600000 vod://321321321
bool SamplePlayer::parseCommandLine(int argc, const char** argv)
{
    return m_options.parse(argc, argv);
}

int SamplePlayer::initialize()
{
    int code = NativePlayer::initialize();
    if (code != 0) {
        return code;
    }

    createPlayers();
    loadCurrentUrl();

    return 0;
}

int SamplePlayer::finalize()
{
    destroyPlayers();
    return NativePlayer::finalize();
}

void SamplePlayer::loadCurrentUrl()
{
    m_state = State::Stopped;
    
    for (auto& player : m_players) {
        int urlIndex = (player == m_players.front()) ? m_urlIndex : m_urlIndex + 1 % m_options.getUrls().size();
        const std::string& url = m_options.getUrls()[urlIndex];
        player->load(url);
    }
}

void SamplePlayer::loadPreviousUrl()
{
    int count = m_options.getUrls().size();
    m_urlIndex--;
    if (m_urlIndex < 0) {
        m_urlIndex = count - 1;
    }
    loadCurrentUrl();
}

void SamplePlayer::loadNextUrl()
{
    int count = m_options.getUrls().size();
    m_urlIndex++;
    m_urlIndex %= count;
    loadCurrentUrl();
}

void SamplePlayer::createPlayers()
{
    if (!PlayerCore::isInitialized()) {
        Configuration config;
        config.deviceId = "team-player-core-testing";
        if (!PlayerCore::initialize(config)) {
            printf("ERROR: Could not initialize PS4 Platform");
            assert(false);
            return;
        }
    }

    StreamPlayer::Configuration config;
    config.defaultInitialQuality = m_options.getDefaultQuality();
    config.initialSeekTime = m_options.getSeekTo();

    assert(m_players.empty());
    size_t numPlayers = m_multiplePlayers ? 2 : 1;
    for (size_t i = 0; i < numPlayers; i++) {
        auto onVideoFrameBuffer = std::bind(&Surface::onVideoFrameBuffer, mp_display->getSurface(i), std::placeholders::_1);
        auto onFrameNeedsReleasing = std::bind(&Surface::onFrameNeedsReleasing, mp_display->getSurface(i), std::placeholders::_1);

        auto player = new StreamPlayer(*this, onVideoFrameBuffer, onFrameNeedsReleasing, config);
        player->getPlayer().setAutoSwitchQuality(!m_options.isAbsDisabled());
        m_players.emplace_back(player);
    }
}

void SamplePlayer::destroyPlayers()
{
    size_t numPlayers = m_multiplePlayers ? 2 : 1;
    for (size_t i = 0; i < numPlayers; i++) {
        auto& player = m_players[i];
        player->pause();
        mp_display->getSurface(i)->onVideoFrameBuffer(nullptr);
        player.reset();
    }

    m_players.clear();
}

void SamplePlayer::reloadPlayers()
{
    destroyPlayers();
    createPlayers();
}

void SamplePlayer::toggleStressTest()
{
    m_stressTest = !m_stressTest;
}

void SamplePlayer::generatePlaySession()
{
    for (auto& player : m_players) {
        player->generatePlaySession();
    }
}

void SamplePlayer::keepPlaySession(bool keep)
{
    for (auto& player : m_players) {
        player->keepPlaySession(keep);
    }
}

void SamplePlayer::togglePlayPause()
{
    if (m_state == State::Playing) {
        m_state = State::Paused;
    } else if (m_state == State::Paused) {
        m_state = State::Playing;
    }

    for (auto& player : m_players) {
        if (m_state == State::Playing) {
            player->play();
        } else if (m_state == State::Paused) {
            player->pause();
        }
    }
}

void SamplePlayer::seekBackwards(twitch::MediaTime time)
{
    for (auto& player : m_players) {
        player->seekBackwards(time);
    }
}

void SamplePlayer::seekForwards(twitch::MediaTime time)
{
    for (auto& player : m_players) {
        player->seekForwards(time);
    }
}

void SamplePlayer::setNextQuality()
{
    for (auto& player : m_players) {
        player->setNextQuality();
    }
}

void SamplePlayer::toggleSplit()
{
    destroyPlayers();

    m_multiplePlayers = !m_multiplePlayers;
    mp_display->toggleSplitView();

    createPlayers();
    loadCurrentUrl();
}

struct ReloadStressTest
{
    bool enabled = false;
    float timeLeftBetweenIteration = TIME_BETWEEN_LOAD;

    static const float TIME_BETWEEN_LOAD;
};

const float ReloadStressTest::TIME_BETWEEN_LOAD = 10.f;
ReloadStressTest reloadStressTest;

void SamplePlayer::updateControls()
{
    auto padContext = getPadContextOfInitialUser();

    if (!padContext) {
        return;
    }

    using namespace sce::SampleUtil::Input;

    // stress test with 'Up'
    if (padContext->isButtonPressed(kButtonUp, kButtonEventPatternAny)) {
        toggleStressTest();
    }
    // Generate a new playSession with 'Down'
    else if (padContext->isButtonPressed(kButtonDown, kButtonEventPatternAny)) {
        generatePlaySession();
    }
    else if (padContext->isButtonPressed(kButtonLeft, kButtonEventPatternAny)) {
        keepPlaySession(true);
    }
    else if (padContext->isButtonPressed(kButtonRight, kButtonEventPatternAny)) {
        keepPlaySession(false);
    }
    // pause/play with 'X'
    else if (padContext->isButtonPressed(kButtonCross, kButtonEventPatternAny)) {
        togglePlayPause();
        // FastForward with R2/Right
    } else if (padContext->isButtonPressed(kButtonR2, kButtonEventPatternAny) || padContext->isButtonPressed(kButtonRight, kButtonEventPatternAny)) {
        seekForwards(MediaTime(5.0f));
        // Rewind with L2/Left
    } else if (padContext->isButtonPressed(kButtonL2, kButtonEventPatternAny) || padContext->isButtonPressed(kButtonLeft, kButtonEventPatternAny)) {
        seekBackwards(MediaTime(5.0f));
        // Change quality with Circle 'O'
    } else if (padContext->isButtonPressed(kButtonCircle, kButtonEventPatternAny)) {
        setNextQuality();
        // Load the previous stream with L1
    } else if (padContext->isButtonPressed(kButtonL1, kButtonEventPatternAny)) {
        loadPreviousUrl();
        // Load the next stream with R1
    } else if (padContext->isButtonPressed(kButtonR1, kButtonEventPatternAny)) {
        loadNextUrl();
        // Reload (Destroy/Create) video Player with Triangle
    } else if (padContext->isButtonPressed(kButtonTriangle, kButtonEventPatternAny)) {
        reloadStressTest.enabled = !reloadStressTest.enabled;
        reloadStressTest.timeLeftBetweenIteration = 0.f;
    } else if (padContext->isButtonPressed(kButtonSquare, kButtonEventPatternAny)) {
        toggleSplit();
    }
}

void SamplePlayer::handleStressTest()
{
    if (!m_stressTest) {
        return;
    }

    // We will send random commands:
    // - Pause/Play
    // - Seek forward and backward
    // - With random timer in between

    bool elapsed = m_stressTestTimer.update(m_dt);
    if (elapsed) {

        int action = rand() % 3;
        if (action == 0) {
            togglePlayPause();
        }
        else if (action == 1) {
            float amount = rand() / static_cast<float>(RAND_MAX) * 5.f;
            seekForwards(MediaTime(amount));
        }
        else if (action == 2) {
            float amount = rand() / static_cast<float>(RAND_MAX) * 5.f;
            seekBackwards(MediaTime(amount));
        }

        // restart the timer
        float newInterval = rand() / static_cast<float>(RAND_MAX) * 3.f;
        m_stressTestTimer.start(newInterval);
    }
}

void SamplePlayer::handleReloadStressTest()
{
    if (reloadStressTest.enabled) {
        reloadStressTest.timeLeftBetweenIteration -= 0.016f;
        if (reloadStressTest.timeLeftBetweenIteration <= 0.f) {
            int choice = rand() % 21;
            if (choice < 10) {
                loadNextUrl();
            }
            else if (choice < 20) {
                loadPreviousUrl();
            }
            else {
                reloadPlayers();
                loadCurrentUrl();
            }

            reloadStressTest.timeLeftBetweenIteration = rand() / static_cast<float>(RAND_MAX) * ReloadStressTest::TIME_BETWEEN_LOAD;
        }
    }
}


void SamplePlayer::updateTimeCounter()
{
    uint64_t timeFreq = sceKernelGetProcessTimeCounterFrequency();
    uint64_t currentTimeCount = sceKernelGetProcessTimeCounter();

    // first tick ?
    if (m_lastTimeCounter == (uint64_t)-1) {
        // assume 60 fps
        m_lastTimeCounter = currentTimeCount - (1.f / 60.f) * timeFreq;
    }

    // how long as it been ?
    m_dt = (currentTimeCount - m_lastTimeCounter) / (float)timeFreq;
    m_lastTimeCounter = currentTimeCount;
}

void SamplePlayer::updateDebugInfo()
{
    // display every second
    static Timer debugInfo(1.f);
    if (!debugInfo.update(m_dt))
        return;

    size_t numPlayers = m_multiplePlayers ? 2 : 1;
    for (size_t i = 0; i < numPlayers; i++) {
        const auto& player = m_players[i]->getPlayer();
        auto position = player.getPosition();
        auto bufferedPosition = player.getBufferedPosition();

        auto bufferSize = bufferedPosition - position;
        printf("Player %zu: Buffer Remaining: %f seconds\n", i, bufferSize.seconds());
    }
}

int SamplePlayer::update()
{
    updateTimeCounter();

    int ret = NativePlayer::update();

    if (ret < 0) {
        return ret;
    }

    // do we want to stop after a given amount of frames ?
    const int numFramesToRun = m_options.getNumFramesToRun();

    if (numFramesToRun != 0 && m_currentFrameCounter++ == numFramesToRun) {
        return 0;
    }

#if PLAYER_RELOAD_STRESS_TEST
    static const MediaTime lowerBound(std::chrono::milliseconds(1));
    static const MediaTime upperBound(std::chrono::seconds(10));
    static MediaTime random = randomDuration(lowerBound, upperBound);

    if (m_state != State::Stopped && m_player->getPosition() > random) {
        reloadPlayers();
        loadNextUrl();
        random = randomDuration(lowerBound, upperBound);
    }
#endif

    updateControls();

    handleStressTest();
    handleReloadStressTest();

    updateDebugInfo();

    return 0;
}

void SamplePlayer::onStreamStarted()
{
    m_state = State::Playing;
}

SamplePlayer g_application;

int main(int argc, const char** argv)
{
    // need to expose ExceptionHandler in our public include
    //ExceptionHandler::registerTerminateHandler(std::this_thread::get_id(), sceUserMainThreadName);
    bool success = g_application.parseCommandLine(argc, argv);

    if (!success) {
        return 1;
    }

    int ret = 0;
    ret = g_application.initialize();

    if (ret < 0) {
        printf("g_application.initialize:0x%08x failed.\n", ret);
        return ret;
    }

    ret = g_application.run();
    if (ret < 0) {
        printf("g_application.run:0x%08x failed.\n", ret);
        return ret;
    }

    ret = g_application.finalize();

    if (ret < 0) {
        printf("g_application.finalize:0x%08x failed.\n", ret);
        return ret;
    }

    return 0;
}
