#pragma once

#include "nativeplayer.hpp"
#include "options.hpp"
#include "streamplayer.hpp"

#include <sampleutil.h>
#include <vector>

namespace twitch {

class Timer {
public:
    Timer(float initialTime)
        : m_initialTime(initialTime)
        , m_remaingTime(m_initialTime)
    {
        assert(m_initialTime >= 0.f);
        assert(m_remaingTime >= 0.f);
    }

    void start(float time)
    {
        m_initialTime = time;
        m_remaingTime = time;

        assert(m_initialTime >= 0.f);
        assert(m_remaingTime >= 0.f);
    }

    bool update(float dt)
    {
        assert(dt >= 0.f);

        m_remaingTime -= dt;
        if (m_remaingTime <= 0.f) {
            m_remaingTime = m_initialTime;
            return true;
        }

        return false;
    }

private:
    float m_initialTime;
    float m_remaingTime;
};

class SamplePlayer : public NativePlayer, public StreamPlayer::Listener {
public:
    SamplePlayer()
        : m_stressTestTimer(1.f)
    {
        srand(time(NULL));
    }

    // Example:
    // nativeplayer -stop-after 3000 -seek-to 2600000 vod://321321321
    bool parseCommandLine(int argc, const char** argv);

    int initialize() override;
    int finalize() override;

    enum class State {
        Stopped,
        Playing,
        Paused,
    };

private:
    int update() override;

    void onStreamStarted() override;

    void loadCurrentUrl();
    void loadPreviousUrl();
    void loadNextUrl();

    void createPlayers();
    void destroyPlayers();
    void reloadPlayers();

    void seekBackwards(twitch::MediaTime time);
    void seekForwards(twitch::MediaTime time);

    void setNextQuality();

    void toggleStressTest();
    
    void generatePlaySession();
    void keepPlaySession(bool keep);

    void togglePlayPause();
    void toggleSplit();

    void updateControls();
    void updateDebugInfo();
    void handleStressTest();
    void handleReloadStressTest();

    void updateTimeCounter();

    Options m_options;

    std::vector<std::unique_ptr<StreamPlayer>> m_players;
    bool m_multiplePlayers = false;
    bool m_stressTest = false;
    Timer m_stressTestTimer;

    int m_urlIndex = 0;

    // The SamplePlayer state
    State m_state = State::Stopped;

    int m_currentFrameCounter = 0;
    uint64_t m_lastTimeCounter = -1;
    float m_dt;
};
}
