#include "streamplayer.hpp"
#include <algorithm>
#include <cassert>
#include <iostream>

using namespace twitch;
using namespace twitch::ps4;

StreamPlayer::StreamPlayer(Listener& listener, OnVideoFrameBuffer onVideoFrameBuffer, OnFrameNeedsReleasing onFrameNeedsReleasing, Configuration config)
    : m_listener(listener)
    , m_player(PlayerCore::create(*this, onVideoFrameBuffer, onFrameNeedsReleasing))
    , m_config(config)
{
    assert(PlayerCore::isInitialized());
    assert(m_player);
}

void StreamPlayer::load(const std::string& url)
{
    m_streamStarted = false;
    m_player->load(url);
}

void StreamPlayer::pause()
{
    if (m_streamStarted) {
        m_player->pause();
    }
}

void StreamPlayer::play()
{
    if (m_streamStarted) {
        m_player->play();
    }
}

void StreamPlayer::seekBackwards(twitch::MediaTime time)
{
    if (!m_streamStarted || !m_player->isSeekable()) {
        return;
    }

    MediaTime currentPosition = m_player->getPosition();
    MediaTime seekPosition = currentPosition - time;
    seekPosition = std::max(seekPosition, MediaTime::zero());
    if (seekPosition != currentPosition) {
        m_player->seekTo(seekPosition);
    }
}

void StreamPlayer::seekForwards(twitch::MediaTime time)
{
    if (!m_streamStarted || !m_player->isSeekable()) {
        return;
    }

    MediaTime currentPosition = m_player->getPosition();
    MediaTime seekPosition = currentPosition + time;
    seekPosition = std::min(seekPosition, m_player->getDuration());
    if (seekPosition != currentPosition) {
        m_player->seekTo(seekPosition);
    }
}

void StreamPlayer::setNextQuality()
{
    if (!m_streamStarted) {
        return;
    }

    const auto& currentQuality = m_player->getQuality();
    const auto& availableQualities = m_player->getQualities();

    auto itr = std::find(availableQualities.begin(), availableQualities.end(), currentQuality);

    // couldn't even find your own quality, what ?
    if (itr == availableQualities.end()) {
        return;
    }

    // cycle to the next
    ++itr;
    if (itr == availableQualities.end()) {
        itr = availableQualities.begin();
    }

    m_player->setQuality(*itr);
}

void StreamPlayer::generatePlaySession()
{
    printf("regeneratePlaySession()");
    m_player->regenerateAnalyticsPlaySession();
}

void StreamPlayer::keepPlaySession(bool keep)
{
    printf("setKeepPlaySession(%s)", keep ? "true" : "false");
    m_player->setKeepAnalyticsPlaySession(keep);
}

void StreamPlayer::setAuthToken(const std::string& token)
{
    m_player->setAuthToken(token);
}

void StreamPlayer::onDurationChanged(MediaTime duration)
{
    printf("onDurationChanged to %f\n", duration.seconds());
}

void StreamPlayer::onError(const Error& error)
{
    printf("onError ErrorSource=%s, MediaResult.value=%s, MediaResult.code=%d, message=%s\n", errorSourceString(error.source), mediaResultString(error.result), error.result.code, error.message.c_str());
}

void StreamPlayer::onQualityChanged(const Quality& quality)
{
    printf("onQualityChanged to %s, bitrate:%d, height:%d, width:%d, group:%s, isDefault:%s, autoSelect:%s\n",
        quality.name.c_str(), quality.bitrate, quality.height, quality.width, quality.group.c_str(), quality.isDefault ? "true" : "false", quality.autoSelect ? "true" : "false");
}

void StreamPlayer::onAnalyticsEvent(const std::string& name, const std::string& properties)
{
    printf("onTracking name=%s, properties=%s\n", name.c_str(), properties.c_str());
}

void StreamPlayer::onSeekCompleted(MediaTime time)
{
    printf("onSeekCompleted %lld us\n", time.microseconds().count());
}

void StreamPlayer::onStateChanged(Player::State state)
{
    m_state = state;
    if (m_state == Player::State::Ready) {
        if (m_config.defaultInitialQuality) {
            setInitialQuality();
        }

        if (m_config.initialSeekTime > 0.0 && m_player->isSeekable()) {
            m_player->seekTo(MediaTime(m_config.initialSeekTime));
        }

        m_player->play();
        m_streamStarted = true;
        m_listener.onStreamStarted();
    }
}

void StreamPlayer::setInitialQuality()
{
    const auto& qualities = m_player->getQualities();
    assert(!qualities.empty());

    auto itr = std::find_if(qualities.begin(), qualities.end(), [](const Quality& quality) { return quality.isDefault; });
    if (itr == qualities.end()) {
        // If the default quality could not be found, use the first quality
        m_player->setQuality(qualities.front());
    } else {
        m_player->setQuality(*itr);
    }
}
